<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Candidato;
use App\Models\Sede;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CandidatoController extends Controller
{
    public function index()
    {
        $candidatos = Candidato::with('sede')->withCount('votos')->get();
        return view('admin.candidatos.index', compact('candidatos'));
    }

    public function create()
    {
        $sedes = Sede::where('activa', true)->get();
        return view('admin.candidatos.create', compact('sedes'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'sede_id' => 'required|exists:sedes,id',
            'nombre' => 'required|string|max:255',
            'tipo' => 'required|in:personero,contralor',
            'propuestas' => 'nullable|string',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $fotoPath = null;
        if ($request->hasFile('foto')) {
            $fotoPath = $request->file('foto')->store('candidatos', 'public');
        }

        Candidato::create([
            'sede_id' => $request->sede_id,
            'nombre' => $request->nombre,
            'tipo' => $request->tipo,
            'propuestas' => $request->propuestas,
            'foto' => $fotoPath,
            'activo' => true,
        ]);

        return redirect()->route('admin.candidatos.index')
            ->with('success', 'Candidato registrado exitosamente.');
    }

    public function edit(Candidato $candidato)
    {
        $sedes = Sede::where('activa', true)->get();
        return view('admin.candidatos.edit', compact('candidato', 'sedes'));
    }

    public function update(Request $request, Candidato $candidato)
    {
        $request->validate([
            'sede_id' => 'required|exists:sedes,id',
            'nombre' => 'required|string|max:255',
            'tipo' => 'required|in:personero,contralor',
            'propuestas' => 'nullable|string',
            'foto' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        if ($request->hasFile('foto')) {
            // Eliminar foto anterior
            if ($candidato->foto) {
                Storage::disk('public')->delete($candidato->foto);
            }
            $candidato->foto = $request->file('foto')->store('candidatos', 'public');
        }

        $candidato->update([
            'sede_id' => $request->sede_id,
            'nombre' => $request->nombre,
            'tipo' => $request->tipo,
            'propuestas' => $request->propuestas,
            'activo' => $request->has('activo'),
        ]);

        return redirect()->route('admin.candidatos.index')
            ->with('success', 'Candidato actualizado exitosamente.');
    }

    public function destroy(Candidato $candidato)
    {
        if ($candidato->foto) {
            Storage::disk('public')->delete($candidato->foto);
        }
        
        $candidato->delete();
        
        return redirect()->route('admin.candidatos.index')
            ->with('success', 'Candidato eliminado exitosamente.');
    }
}
