<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Candidato;
use App\Models\Grado;
use App\Models\Sede;
use App\Models\Setting;
use App\Models\User;
use App\Models\Voto;
use Illuminate\Http\Request;

class DashboardController extends Controller
{
    public function index()
    {
        $totalEstudiantes = User::where('role', 'estudiante')->count();
        $estudiantesHabilitados = User::where('role', 'estudiante')->where('habilitado', true)->count();
        $estudiantesQueVotaron = User::where('role', 'estudiante')
            ->where(function($q) {
                $q->where('ha_votado_personero', true)->orWhere('ha_votado_contralor', true);
            })->count();
        
        $totalVotosPersonero = Voto::whereHas('candidato', fn($q) => $q->where('tipo', 'personero'))->count();
        $totalVotosContralor = Voto::whereHas('candidato', fn($q) => $q->where('tipo', 'contralor'))->count();
        
        // Resultados por sede
        $sedes = Sede::with(['candidatos' => function ($query) {
            $query->withCount('votos');
        }, 'grados'])->where('activa', true)->get();

        $resultadosPorSede = [];
        foreach ($sedes as $sede) {
            $estudiantesSede = User::where('role', 'estudiante')
                ->whereHas('grado', fn($q) => $q->where('sede_id', $sede->id))
                ->count();
            
            $votaronSede = User::where('role', 'estudiante')
                ->whereHas('grado', fn($q) => $q->where('sede_id', $sede->id))
                ->where(function($q) {
                    $q->where('ha_votado_personero', true)->orWhere('ha_votado_contralor', true);
                })->count();

            $personeros = $sede->candidatos->where('tipo', 'personero')->sortByDesc('votos_count');
            $contralores = $sede->candidatos->where('tipo', 'contralor')->sortByDesc('votos_count');
            
            $resultadosPorSede[$sede->id] = [
                'sede' => $sede,
                'estudiantes' => $estudiantesSede,
                'votaron' => $votaronSede,
                'porcentaje' => $estudiantesSede > 0 ? round(($votaronSede / $estudiantesSede) * 100, 1) : 0,
                'personeros' => $personeros,
                'contralores' => $contralores,
            ];
        }

        // Resultados por grado
        $grados = Grado::with('sede')->get();
        $resultadosPorGrado = [];
        foreach ($grados as $grado) {
            $estudiantesGrado = User::where('role', 'estudiante')->where('grado_id', $grado->id)->count();
            $votaronGrado = User::where('role', 'estudiante')
                ->where('grado_id', $grado->id)
                ->where(function($q) {
                    $q->where('ha_votado_personero', true)->orWhere('ha_votado_contralor', true);
                })->count();
            
            if ($estudiantesGrado > 0) {
                $resultadosPorGrado[] = [
                    'grado' => $grado,
                    'sede' => $grado->sede->nombre,
                    'estudiantes' => $estudiantesGrado,
                    'votaron' => $votaronGrado,
                    'porcentaje' => round(($votaronGrado / $estudiantesGrado) * 100, 1),
                ];
            }
        }

        $votacionAbierta = Setting::votacionAbierta();

        return view('admin.dashboard', compact(
            'totalEstudiantes',
            'estudiantesHabilitados',
            'estudiantesQueVotaron',
            'totalVotosPersonero',
            'totalVotosContralor',
            'sedes',
            'resultadosPorSede',
            'resultadosPorGrado',
            'votacionAbierta'
        ));
    }

    public function toggleVotacion()
    {
        $actual = Setting::votacionAbierta();
        Setting::set('votacion_abierta', $actual ? 'false' : 'true');
        
        $mensaje = $actual ? 'Votación cerrada. Los resultados ahora son públicos.' : 'Votación abierta. Los estudiantes pueden votar.';
        
        return redirect()->route('admin.dashboard')->with('success', $mensaje);
    }

    public function resultados(Request $request)
    {
        $sedeId = $request->get('sede_id');
        
        $query = Candidato::withCount('votos');
        
        if ($sedeId) {
            $query->where('sede_id', $sedeId);
        }
        
        $candidatos = $query->get();
        $sedes = Sede::where('activa', true)->get();
        
        $personeros = $candidatos->where('tipo', 'personero')->sortByDesc('votos_count');
        $contralores = $candidatos->where('tipo', 'contralor')->sortByDesc('votos_count');
        
        // Datos para gráficas
        $chartDataPersoneros = [
            'labels' => $personeros->pluck('nombre')->toArray(),
            'data' => $personeros->pluck('votos_count')->toArray(),
        ];
        
        $chartDataContralores = [
            'labels' => $contralores->pluck('nombre')->toArray(),
            'data' => $contralores->pluck('votos_count')->toArray(),
        ];

        return view('admin.resultados', compact(
            'personeros',
            'contralores',
            'sedes',
            'sedeId',
            'chartDataPersoneros',
            'chartDataContralores'
        ));
    }
}
