<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Grado;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class EstudianteController extends Controller
{
    public function index(Request $request)
    {
        $query = User::where('role', 'estudiante')->with('grado.sede');
        
        if ($request->has('grado_id') && $request->grado_id) {
            $query->where('grado_id', $request->grado_id);
        }
        
        if ($request->has('habilitado') && $request->habilitado !== '') {
            $query->where('habilitado', $request->habilitado);
        }
        
        $estudiantes = $query->get();
        $grados = Grado::with('sede')->get();
        
        return view('admin.estudiantes.index', compact('estudiantes', 'grados'));
    }

    public function create()
    {
        $grados = Grado::with('sede')->get();
        $sedes = \App\Models\Sede::where('activa', true)->get();
        return view('admin.estudiantes.create', compact('grados', 'sedes'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'documento' => 'required|string|unique:users,documento',
            'grado_id' => 'required|exists:grados,id',
            'email' => 'nullable|email|unique:users,email',
        ]);

        User::create([
            'name' => $request->name,
            'documento' => $request->documento,
            'email' => $request->email ?? $request->documento . '@estudiante.local',
            'password' => Hash::make($request->documento),
            'grado_id' => $request->grado_id,
            'role' => 'estudiante',
            'habilitado' => $request->has('habilitado'),
        ]);

        return redirect()->route('admin.estudiantes.index')
            ->with('success', 'Estudiante registrado exitosamente.');
    }

    public function edit(User $estudiante)
    {
        $grados = Grado::with('sede')->get();
        return view('admin.estudiantes.edit', compact('estudiante', 'grados'));
    }

    public function update(Request $request, User $estudiante)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'documento' => 'required|string|unique:users,documento,' . $estudiante->id,
            'grado_id' => 'required|exists:grados,id',
            'email' => 'nullable|email|unique:users,email,' . $estudiante->id,
        ]);

        $estudiante->update([
            'name' => $request->name,
            'documento' => $request->documento,
            'email' => $request->email,
            'grado_id' => $request->grado_id,
            'habilitado' => $request->has('habilitado'),
        ]);

        return redirect()->route('admin.estudiantes.index')
            ->with('success', 'Estudiante actualizado exitosamente.');
    }

    public function destroy(User $estudiante)
    {
        $estudiante->delete();
        
        return redirect()->route('admin.estudiantes.index')
            ->with('success', 'Estudiante eliminado exitosamente.');
    }

    public function habilitarTodos()
    {
        User::where('role', 'estudiante')->update(['habilitado' => true]);
        
        return redirect()->route('admin.estudiantes.index')
            ->with('success', 'Todos los estudiantes han sido habilitados.');
    }

    public function resetearVotos()
    {
        User::where('role', 'estudiante')->update([
            'ha_votado_personero' => false,
            'ha_votado_contralor' => false,
        ]);
        
        return redirect()->route('admin.estudiantes.index')
            ->with('success', 'Votos de todos los estudiantes han sido reseteados.');
    }
}
