<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Candidato;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class CandidatoApiController extends Controller
{
    public function index(Request $request)
    {
        $query = Candidato::with('sede')->withCount('votos');

        if ($request->has('sede_id')) {
            $query->where('sede_id', $request->sede_id);
        }

        if ($request->has('tipo')) {
            $query->where('tipo', $request->tipo);
        }

        $candidatos = $query->get();
        return response()->json($candidatos);
    }

    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'tipo' => 'required|in:personero,contralor',
            'sede_id' => 'required|exists:sedes,id',
            'propuestas' => 'nullable|string',
            'foto' => 'nullable|image|max:2048',
            'activo' => 'boolean',
        ]);

        $data = $request->only(['nombre', 'tipo', 'sede_id', 'propuestas', 'activo']);

        if ($request->hasFile('foto')) {
            $data['foto'] = $request->file('foto')->store('candidatos', 'public');
        }

        $candidato = Candidato::create($data);

        return response()->json($candidato->load('sede'), 201);
    }

    public function show(Candidato $candidato)
    {
        return response()->json($candidato->load('sede')->loadCount('votos'));
    }

    public function update(Request $request, Candidato $candidato)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'tipo' => 'required|in:personero,contralor',
            'sede_id' => 'required|exists:sedes,id',
            'propuestas' => 'nullable|string',
            'foto' => 'nullable|image|max:2048',
            'activo' => 'boolean',
        ]);

        $data = $request->only(['nombre', 'tipo', 'sede_id', 'propuestas', 'activo']);

        if ($request->hasFile('foto')) {
            // Delete old photo
            if ($candidato->foto) {
                Storage::disk('public')->delete($candidato->foto);
            }
            $data['foto'] = $request->file('foto')->store('candidatos', 'public');
        }

        $candidato->update($data);

        return response()->json($candidato->load('sede'));
    }

    public function destroy(Candidato $candidato)
    {
        if ($candidato->foto) {
            Storage::disk('public')->delete($candidato->foto);
        }

        $candidato->delete();
        return response()->json(['message' => 'Candidato eliminado correctamente']);
    }
}
