<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;

class EstudianteApiController extends Controller
{
    public function index(Request $request)
    {
        $query = User::where('role', 'estudiante')->with('grado.sede');

        if ($request->has('grado_id')) {
            $query->where('grado_id', $request->grado_id);
        }

        if ($request->has('sede_id')) {
            $query->whereHas('grado', function ($q) use ($request) {
                $q->where('sede_id', $request->sede_id);
            });
        }

        if ($request->has('habilitado')) {
            $query->where('habilitado', $request->boolean('habilitado'));
        }

        $estudiantes = $query->get();
        return response()->json($estudiantes);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:users',
            'documento' => 'required|string|unique:users',
            'grado_id' => 'required|exists:grados,id',
            'habilitado' => 'boolean',
        ]);

        // Generar email automático si no se proporciona
        $email = $request->email ?? $request->documento . '@estudiante.local';

        $estudiante = User::create([
            'name' => $request->name,
            'email' => $email,
            'documento' => $request->documento,
            'grado_id' => $request->grado_id,
            'password' => Hash::make($request->documento), // Contraseña = documento
            'role' => 'estudiante',
            'habilitado' => $request->boolean('habilitado', true),
        ]);

        return response()->json($estudiante->load('grado.sede'), 201);
    }

    public function show(User $estudiante)
    {
        if ($estudiante->role !== 'estudiante') {
            return response()->json(['error' => 'Usuario no es estudiante'], 404);
        }

        return response()->json($estudiante->load('grado.sede'));
    }

    public function update(Request $request, User $estudiante)
    {
        if ($estudiante->role !== 'estudiante') {
            return response()->json(['error' => 'Usuario no es estudiante'], 404);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'documento' => 'required|string|unique:users,documento,' . $estudiante->id,
            'grado_id' => 'required|exists:grados,id',
            'habilitado' => 'boolean',
        ]);

        $estudiante->update($request->only(['name', 'documento', 'grado_id', 'habilitado']));

        return response()->json($estudiante->load('grado.sede'));
    }

    public function destroy(User $estudiante)
    {
        if ($estudiante->role !== 'estudiante') {
            return response()->json(['error' => 'Usuario no es estudiante'], 404);
        }

        $estudiante->delete();
        return response()->json(['message' => 'Estudiante eliminado correctamente']);
    }

    public function habilitarTodos()
    {
        User::where('role', 'estudiante')->update(['habilitado' => true]);
        return response()->json(['message' => 'Todos los estudiantes han sido habilitados']);
    }

    public function resetearVotos()
    {
        User::where('role', 'estudiante')->update([
            'ha_votado_personero' => false,
            'ha_votado_contralor' => false,
        ]);

        return response()->json(['message' => 'Votos de estudiantes reiniciados']);
    }

    /**
     * Export students to CSV
     */
    public function export(Request $request)
    {
        $query = User::where('role', 'estudiante')->with('grado.sede');

        if ($request->has('sede_id')) {
            $query->whereHas('grado', function ($q) use ($request) {
                $q->where('sede_id', $request->sede_id);
            });
        }

        if ($request->has('grado_id')) {
            $query->where('grado_id', $request->grado_id);
        }

        $estudiantes = $query->get();

        $csv = "documento,nombre,grado,sede,habilitado,ha_votado_personero,ha_votado_contralor\n";
        
        foreach ($estudiantes as $e) {
            $csv .= implode(',', [
                $e->documento,
                '"' . str_replace('"', '""', $e->name) . '"',
                '"' . ($e->grado?->nombre ?? '') . '"',
                '"' . ($e->grado?->sede?->nombre ?? '') . '"',
                $e->habilitado ? '1' : '0',
                $e->ha_votado_personero ? '1' : '0',
                $e->ha_votado_contralor ? '1' : '0',
            ]) . "\n";
        }

        return response($csv, 200, [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => 'attachment; filename="estudiantes_' . date('Y-m-d') . '.csv"',
        ]);
    }

    /**
     * Import students from CSV
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|file|mimes:csv,txt|max:5120',
            'grado_id' => 'required|exists:grados,id',
        ]);

        $file = $request->file('file');
        $content = file_get_contents($file->getPathname());
        $lines = explode("\n", trim($content));
        
        // Remove header
        array_shift($lines);
        
        $imported = 0;
        $errors = [];
        $skipped = 0;

        foreach ($lines as $index => $line) {
            $line = trim($line);
            if (empty($line)) continue;

            // Parse CSV line
            $data = str_getcsv($line);
            
            if (count($data) < 2) {
                $errors[] = "Línea " . ($index + 2) . ": formato inválido";
                continue;
            }

            $documento = trim($data[0]);
            $nombre = trim($data[1]);

            if (empty($documento) || empty($nombre)) {
                $errors[] = "Línea " . ($index + 2) . ": documento o nombre vacío";
                continue;
            }

            // Check if student already exists
            $existing = User::where('documento', $documento)->first();
            if ($existing) {
                $skipped++;
                continue;
            }

            try {
                User::create([
                    'name' => $nombre,
                    'email' => $documento . '@estudiante.local',
                    'documento' => $documento,
                    'grado_id' => $request->grado_id,
                    'password' => Hash::make($documento),
                    'role' => 'estudiante',
                    'habilitado' => true,
                ]);
                $imported++;
            } catch (\Exception $e) {
                $errors[] = "Línea " . ($index + 2) . ": " . $e->getMessage();
            }
        }

        return response()->json([
            'message' => "Importación completada",
            'imported' => $imported,
            'skipped' => $skipped,
            'errors' => array_slice($errors, 0, 10), // Only first 10 errors
            'total_errors' => count($errors),
        ]);
    }
}
