<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Grado;
use App\Models\Sede;
use Illuminate\Http\Request;

class GradoApiController extends Controller
{
    public function index()
    {
        $grados = Grado::with('sede')->withCount('estudiantes')->get();
        return response()->json($grados);
    }

    public function store(Request $request)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'sede_id' => 'required|exists:sedes,id',
        ]);

        // Extraer nivel del nombre (ej: "Grado 6°" -> 6, "6-1" -> 6)
        $nombre = $request->nombre;
        preg_match('/(\d+)/', $nombre, $matches);
        $nivel = isset($matches[1]) ? (int) $matches[1] : 0;

        $grado = Grado::create([
            'nombre' => $nombre,
            'sede_id' => $request->sede_id,
            'nivel' => $nivel,
        ]);

        return response()->json($grado->load('sede'), 201);
    }

    public function show(Grado $grado)
    {
        return response()->json($grado->load(['sede', 'estudiantes']));
    }

    public function update(Request $request, Grado $grado)
    {
        $request->validate([
            'nombre' => 'required|string|max:255',
            'sede_id' => 'required|exists:sedes,id',
        ]);

        $grado->update($request->only(['nombre', 'sede_id']));

        return response()->json($grado->load('sede'));
    }

    public function destroy(Grado $grado)
    {
        $grado->delete();
        return response()->json(['message' => 'Grado eliminado correctamente']);
    }

    public function getBySede(Sede $sede)
    {
        $grados = Grado::where('sede_id', $sede->id)->get();
        return response()->json($grados);
    }
}
