<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Candidato;
use App\Models\Sede;
use App\Models\Setting;
use App\Models\User;
use App\Models\Voto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class VotacionController extends Controller
{
    /**
     * Get all active sedes
     */
    public function sedes()
    {
        $sedes = Sede::where('activa', true)->get();
        $votacionAbierta = Setting::votacionAbierta();

        return response()->json([
            'sedes' => $sedes,
            'votacionAbierta' => $votacionAbierta,
            'institucion' => [
                'nombre' => Setting::get('institucion_nombre', 'I.E. Las Huacas'),
                'logo' => Setting::get('institucion_logo'),
            ],
        ]);
    }

    /**
     * Get sede details with candidates
     */
    public function sede(Sede $sede)
    {
        if (!$sede->activa) {
            return response()->json(['error' => 'Esta sede no está disponible.'], 404);
        }

        $votacionAbierta = Setting::votacionAbierta();

        $personeros = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'personero')
            ->where('activo', true)
            ->get();

        $contralores = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'contralor')
            ->where('activo', true)
            ->get();

        return response()->json([
            'sede' => $sede,
            'personeros' => $personeros,
            'contralores' => $contralores,
            'votacionAbierta' => $votacionAbierta,
        ]);
    }

    /**
     * Verify student by document number
     */
    public function verificar(Request $request, Sede $sede)
    {
        $request->validate([
            'documento' => 'required|string',
        ]);

        if (!Setting::votacionAbierta()) {
            return response()->json(['error' => 'La votación ha sido cerrada.'], 403);
        }

        $estudiante = User::where('documento', $request->documento)
            ->where('role', 'estudiante')
            ->with('grado.sede')
            ->first();

        if (!$estudiante) {
            return response()->json(['error' => 'No se encontró ningún estudiante con ese número de documento.'], 404);
        }

        if (!$estudiante->habilitado) {
            return response()->json(['error' => 'No estás habilitado para votar. Contacta al administrador.'], 403);
        }

        // Verify student belongs to this sede
        $sedeIdEstudiante = $estudiante->grado?->sede_id;
        if (!$sedeIdEstudiante || (int)$sedeIdEstudiante !== (int)$sede->id) {
            return response()->json(['error' => 'No perteneces a esta sede. Ingresa al link de tu sede correspondiente.'], 403);
        }

        return response()->json([
            'estudiante' => [
                'id' => $estudiante->id,
                'name' => $estudiante->name,
                'documento' => $estudiante->documento,
                'grado' => $estudiante->grado?->nombre,
                'sede' => $estudiante->grado?->sede?->nombre,
                'ha_votado_personero' => $estudiante->ha_votado_personero,
                'ha_votado_contralor' => $estudiante->ha_votado_contralor,
            ],
            'token' => encrypt($estudiante->id . '|' . $sede->id . '|' . now()->timestamp),
        ]);
    }

    /**
     * Get candidates for voting (after verification)
     */
    public function candidatos(Request $request, Sede $sede)
    {
        $request->validate([
            'token' => 'required|string',
        ]);

        try {
            $data = explode('|', decrypt($request->token));
            $estudianteId = $data[0];
            $sedeId = $data[1];
            $timestamp = $data[2];

            // Token expires after 30 minutes
            if (now()->timestamp - $timestamp > 1800) {
                return response()->json(['error' => 'Sesión expirada. Por favor verifica tu documento nuevamente.'], 403);
            }

            if ((int)$sedeId !== (int)$sede->id) {
                return response()->json(['error' => 'Token inválido para esta sede.'], 403);
            }

            $estudiante = User::find($estudianteId);
            if (!$estudiante || !$estudiante->habilitado) {
                return response()->json(['error' => 'Estudiante no encontrado o no habilitado.'], 403);
            }
        } catch (\Exception $e) {
            return response()->json(['error' => 'Token inválido.'], 403);
        }

        $personeros = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'personero')
            ->where('activo', true)
            ->get();

        $contralores = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'contralor')
            ->where('activo', true)
            ->get();

        return response()->json([
            'estudiante' => [
                'id' => $estudiante->id,
                'name' => $estudiante->name,
                'ha_votado_personero' => $estudiante->ha_votado_personero,
                'ha_votado_contralor' => $estudiante->ha_votado_contralor,
            ],
            'personeros' => $personeros,
            'contralores' => $contralores,
        ]);
    }

    /**
     * Register vote
     */
    public function votar(Request $request, Sede $sede)
    {
        $request->validate([
            'token' => 'required|string',
            'personero_id' => 'nullable|exists:candidatos,id',
            'contralor_id' => 'nullable|exists:candidatos,id',
        ]);

        if (!Setting::votacionAbierta()) {
            return response()->json(['error' => 'La votación ha sido cerrada.'], 403);
        }

        try {
            $data = explode('|', decrypt($request->token));
            $estudianteId = $data[0];
            $sedeId = $data[1];
            $timestamp = $data[2];

            if (now()->timestamp - $timestamp > 1800 || (int)$sedeId !== (int)$sede->id) {
                return response()->json(['error' => 'Sesión expirada o inválida.'], 403);
            }

            $estudiante = User::find($estudianteId);
            if (!$estudiante || !$estudiante->habilitado) {
                return response()->json(['error' => 'No habilitado para votar.'], 403);
            }
        } catch (\Exception $e) {
            return response()->json(['error' => 'Token inválido.'], 403);
        }

        DB::beginTransaction();

        try {
            if ($request->personero_id && !$estudiante->ha_votado_personero) {
                $personero = Candidato::where('id', $request->personero_id)
                    ->where('sede_id', $sede->id)
                    ->where('tipo', 'personero')
                    ->firstOrFail();

                // Verificar si ya existe un voto para personero
                $yaVotoPersonero = Voto::whereHas('candidato', function ($q) use ($sede) {
                    $q->where('sede_id', $sede->id)->where('tipo', 'personero');
                })->where('user_id', $estudiante->id)->exists();

                if (!$yaVotoPersonero) {
                    Voto::create([
                        'user_id' => $estudiante->id,
                        'candidato_id' => $personero->id,
                    ]);
                }

                $estudiante->update(['ha_votado_personero' => true]);
            }

            if ($request->contralor_id && !$estudiante->ha_votado_contralor) {
                $contralor = Candidato::where('id', $request->contralor_id)
                    ->where('sede_id', $sede->id)
                    ->where('tipo', 'contralor')
                    ->firstOrFail();

                // Verificar si ya existe un voto para contralor
                $yaVotoContralor = Voto::whereHas('candidato', function ($q) use ($sede) {
                    $q->where('sede_id', $sede->id)->where('tipo', 'contralor');
                })->where('user_id', $estudiante->id)->exists();

                if (!$yaVotoContralor) {
                    Voto::create([
                        'user_id' => $estudiante->id,
                        'candidato_id' => $contralor->id,
                    ]);
                }

                $estudiante->update(['ha_votado_contralor' => true]);
            }

            DB::commit();

            return response()->json([
                'message' => '¡Tu voto ha sido registrado exitosamente!',
                'ha_votado_personero' => $estudiante->fresh()->ha_votado_personero,
                'ha_votado_contralor' => $estudiante->fresh()->ha_votado_contralor,
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'error' => 'Error al registrar tu voto. Por favor intenta de nuevo.'
            ], 500);
        }
    }

    /**
     * Get public results (only when voting is closed)
     */
    public function resultadosPublicos()
    {
        $votacionAbierta = Setting::votacionAbierta();

        if ($votacionAbierta) {
            return response()->json(['error' => 'Los resultados estarán disponibles cuando la votación cierre.'], 403);
        }

        $sedes = Sede::where('activa', true)->get();
        $resultados = [];

        foreach ($sedes as $sede) {
            $personeros = Candidato::where('sede_id', $sede->id)
                ->where('tipo', 'personero')
                ->withCount('votos')
                ->orderByDesc('votos_count')
                ->get();

            $contralores = Candidato::where('sede_id', $sede->id)
                ->where('tipo', 'contralor')
                ->withCount('votos')
                ->orderByDesc('votos_count')
                ->get();

            $resultados[] = [
                'sede' => $sede,
                'personeros' => $personeros,
                'contralores' => $contralores,
            ];
        }

        return response()->json([
            'resultados' => $resultados,
            'votacionAbierta' => false,
        ]);
    }
}
