<?php

namespace App\Http\Controllers;

use App\Models\Candidato;
use App\Models\Voto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class VotacionController extends Controller
{
    public function index()
    {
        $user = Auth::user();
        
        // Verificar si el usuario puede votar
        if (!$user->puedeVotar()) {
            return redirect()->route('dashboard')
                ->with('error', 'No estás habilitado para votar.');
        }
        
        $sede = $user->getSede();
        
        if (!$sede) {
            return redirect()->route('dashboard')
                ->with('error', 'No tienes una sede asignada.');
        }
        
        $personeros = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'personero')
            ->where('activo', true)
            ->get();
            
        $contralores = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'contralor')
            ->where('activo', true)
            ->get();
        
        return view('votacion.index', compact(
            'user',
            'sede',
            'personeros',
            'contralores'
        ));
    }

    public function votar(Request $request)
    {
        $user = Auth::user();
        
        if (!$user->puedeVotar()) {
            return redirect()->route('dashboard')
                ->with('error', 'No estás habilitado para votar.');
        }
        
        $request->validate([
            'personero_id' => 'nullable|exists:candidatos,id',
            'contralor_id' => 'nullable|exists:candidatos,id',
        ]);
        
        $sede = $user->getSede();
        
        DB::beginTransaction();
        
        try {
            // Votar por personero
            if ($request->personero_id && !$user->yaVotoPersonero()) {
                $personero = Candidato::where('id', $request->personero_id)
                    ->where('sede_id', $sede->id)
                    ->where('tipo', 'personero')
                    ->firstOrFail();
                
                Voto::create([
                    'user_id' => $user->id,
                    'candidato_id' => $personero->id,
                ]);
                
                $user->update(['ha_votado_personero' => true]);
            }
            
            // Votar por contralor
            if ($request->contralor_id && !$user->yaVotoContralor()) {
                $contralor = Candidato::where('id', $request->contralor_id)
                    ->where('sede_id', $sede->id)
                    ->where('tipo', 'contralor')
                    ->firstOrFail();
                
                Voto::create([
                    'user_id' => $user->id,
                    'candidato_id' => $contralor->id,
                ]);
                
                $user->update(['ha_votado_contralor' => true]);
            }
            
            DB::commit();
            
            return redirect()->route('votacion.confirmacion')
                ->with('success', '¡Tu voto ha sido registrado exitosamente!');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Ocurrió un error al registrar tu voto. Intenta de nuevo.');
        }
    }

    public function confirmacion()
    {
        $user = Auth::user();
        
        return view('votacion.confirmacion', compact('user'));
    }
}
