<?php

namespace App\Http\Controllers;

use App\Models\Candidato;
use App\Models\User;
use App\Models\Voto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class VotacionPublicaController extends Controller
{
    /**
     * Verificar estudiante por documento
     */
    public function verificar(Request $request)
    {
        $request->validate([
            'documento' => 'required|string',
        ]);

        $estudiante = User::where('documento', $request->documento)
            ->where('role', 'estudiante')
            ->with('grado.sede')
            ->first();

        if (!$estudiante) {
            return back()->with('error', 'No se encontró ningún estudiante con ese número de documento.');
        }

        if (!$estudiante->habilitado) {
            return back()->with('error', 'No estás habilitado para votar. Contacta al administrador.');
        }

        // Guardar el ID del estudiante en la sesión
        session(['estudiante_id' => $estudiante->id]);

        return redirect()->route('votar.mostrar');
    }

    /**
     * Mostrar pantalla de votación
     */
    public function mostrar()
    {
        $estudianteId = session('estudiante_id');

        if (!$estudianteId) {
            return redirect()->route('home')->with('error', 'Por favor ingresa tu número de documento.');
        }

        $estudiante = User::with('grado.sede')->find($estudianteId);

        if (!$estudiante || !$estudiante->habilitado) {
            session()->forget('estudiante_id');
            return redirect()->route('home')->with('error', 'Sesión inválida. Por favor intenta de nuevo.');
        }

        $sede = $estudiante->grado?->sede;

        if (!$sede) {
            return redirect()->route('home')->with('error', 'No tienes una sede asignada.');
        }

        $personeros = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'personero')
            ->where('activo', true)
            ->get();

        $contralores = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'contralor')
            ->where('activo', true)
            ->get();

        return view('votacion.publica', compact(
            'estudiante',
            'sede',
            'personeros',
            'contralores'
        ));
    }

    /**
     * Registrar voto
     */
    public function votar(Request $request)
    {
        $estudianteId = session('estudiante_id');

        if (!$estudianteId) {
            return redirect()->route('home')->with('error', 'Sesión expirada. Por favor ingresa tu documento nuevamente.');
        }

        $estudiante = User::find($estudianteId);

        if (!$estudiante || !$estudiante->habilitado) {
            session()->forget('estudiante_id');
            return redirect()->route('home')->with('error', 'No estás habilitado para votar.');
        }

        $request->validate([
            'personero_id' => 'nullable|exists:candidatos,id',
            'contralor_id' => 'nullable|exists:candidatos,id',
        ]);

        $sede = $estudiante->grado?->sede;

        DB::beginTransaction();

        try {
            // Votar por personero
            if ($request->personero_id && !$estudiante->ha_votado_personero) {
                $personero = Candidato::where('id', $request->personero_id)
                    ->where('sede_id', $sede->id)
                    ->where('tipo', 'personero')
                    ->firstOrFail();

                Voto::create([
                    'user_id' => $estudiante->id,
                    'candidato_id' => $personero->id,
                ]);

                $estudiante->update(['ha_votado_personero' => true]);
            }

            // Votar por contralor
            if ($request->contralor_id && !$estudiante->ha_votado_contralor) {
                $contralor = Candidato::where('id', $request->contralor_id)
                    ->where('sede_id', $sede->id)
                    ->where('tipo', 'contralor')
                    ->firstOrFail();

                Voto::create([
                    'user_id' => $estudiante->id,
                    'candidato_id' => $contralor->id,
                ]);

                $estudiante->update(['ha_votado_contralor' => true]);
            }

            DB::commit();

            // Limpiar sesión después de votar
            session()->forget('estudiante_id');

            return redirect()->route('votar.confirmacion');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Ocurrió un error al registrar tu voto. Intenta de nuevo.');
        }
    }

    /**
     * Confirmación de voto
     */
    public function confirmacion()
    {
        return view('votacion.confirmacion-publica');
    }
}
