<?php

namespace App\Http\Controllers;

use App\Models\Candidato;
use App\Models\Sede;
use App\Models\Setting;
use App\Models\User;
use App\Models\Voto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class VotacionSedeController extends Controller
{
    /**
     * Mostrar página de votación para una sede específica
     */
    public function index(Sede $sede)
    {
        // Verificar si la votación está cerrada
        if (!Setting::votacionAbierta()) {
            return redirect()->route('home')->with('error', 'La votación ha sido cerrada.');
        }

        if (!$sede->activa) {
            abort(404, 'Esta sede no está disponible para votación.');
        }

        $personeros = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'personero')
            ->where('activo', true)
            ->get();

        $contralores = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'contralor')
            ->where('activo', true)
            ->get();

        return view('votacion.sede', compact('sede', 'personeros', 'contralores'));
    }

    /**
     * Verificar estudiante y mostrar formulario de votación
     */
    public function verificar(Request $request, Sede $sede)
    {
        $request->validate([
            'documento' => 'required|string',
        ]);

        $estudiante = User::where('documento', $request->documento)
            ->where('role', 'estudiante')
            ->with('grado.sede')
            ->first();

        if (!$estudiante) {
            return back()->with('error', 'No se encontró ningún estudiante con ese número de documento.');
        }

        if (!$estudiante->habilitado) {
            return back()->with('error', 'No estás habilitado para votar. Contacta al administrador.');
        }

        // Verificar que el estudiante pertenece a esta sede
        $sedeIdEstudiante = $estudiante->grado?->sede_id;
        if (!$sedeIdEstudiante || (int)$sedeIdEstudiante !== (int)$sede->id) {
            return back()->with('error', 'No perteneces a esta sede. Ingresa al link de tu sede correspondiente.');
        }

        // Guardar en sesión
        session(['estudiante_id' => $estudiante->id, 'sede_id' => $sede->id]);

        return redirect()->route('sede.votar.mostrar', $sede);
    }

    /**
     * Mostrar formulario de votación
     */
    public function mostrar(Sede $sede)
    {
        $estudianteId = session('estudiante_id');
        $sedeIdSesion = session('sede_id');

        if (!$estudianteId || $sedeIdSesion !== $sede->id) {
            return redirect()->route('sede.votar', $sede)->with('error', 'Por favor ingresa tu documento.');
        }

        $estudiante = User::with('grado.sede')->find($estudianteId);

        if (!$estudiante || !$estudiante->habilitado) {
            session()->forget(['estudiante_id', 'sede_id']);
            return redirect()->route('sede.votar', $sede)->with('error', 'Sesión inválida.');
        }

        $personeros = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'personero')
            ->where('activo', true)
            ->get();

        $contralores = Candidato::where('sede_id', $sede->id)
            ->where('tipo', 'contralor')
            ->where('activo', true)
            ->get();

        return view('votacion.sede-votar', compact('sede', 'estudiante', 'personeros', 'contralores'));
    }

    /**
     * Registrar voto
     */
    public function votar(Request $request, Sede $sede)
    {
        // Verificar si la votación está cerrada
        if (!Setting::votacionAbierta()) {
            return redirect()->route('home')->with('error', 'La votación ha sido cerrada.');
        }

        $estudianteId = session('estudiante_id');

        if (!$estudianteId) {
            return redirect()->route('sede.votar', $sede)->with('error', 'Sesión expirada.');
        }

        $estudiante = User::find($estudianteId);

        if (!$estudiante || !$estudiante->habilitado) {
            session()->forget(['estudiante_id', 'sede_id']);
            return redirect()->route('sede.votar', $sede)->with('error', 'No habilitado para votar.');
        }

        $request->validate([
            'personero_id' => 'nullable|exists:candidatos,id',
            'contralor_id' => 'nullable|exists:candidatos,id',
        ]);

        DB::beginTransaction();

        try {
            if ($request->personero_id && !$estudiante->ha_votado_personero) {
                $personero = Candidato::where('id', $request->personero_id)
                    ->where('sede_id', $sede->id)
                    ->where('tipo', 'personero')
                    ->firstOrFail();

                Voto::create([
                    'user_id' => $estudiante->id,
                    'candidato_id' => $personero->id,
                ]);

                $estudiante->update(['ha_votado_personero' => true]);
            }

            if ($request->contralor_id && !$estudiante->ha_votado_contralor) {
                $contralor = Candidato::where('id', $request->contralor_id)
                    ->where('sede_id', $sede->id)
                    ->where('tipo', 'contralor')
                    ->firstOrFail();

                Voto::create([
                    'user_id' => $estudiante->id,
                    'candidato_id' => $contralor->id,
                ]);

                $estudiante->update(['ha_votado_contralor' => true]);
            }

            DB::commit();
            session()->forget(['estudiante_id', 'sede_id']);

            return redirect()->route('sede.votar.confirmacion', $sede);

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error al registrar tu voto. Intenta de nuevo.');
        }
    }

    /**
     * Confirmación de voto
     */
    public function confirmacion(Sede $sede)
    {
        return view('votacion.sede-confirmacion', compact('sede'));
    }
}
