<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class Candidato extends Model
{
    use HasFactory;

    protected $fillable = [
        'sede_id',
        'nombre',
        'foto',
        'tipo',
        'propuestas',
        'activo',
    ];

    protected $casts = [
        'activo' => 'boolean',
    ];

    /**
     * Sede del candidato
     */
    public function sede(): BelongsTo
    {
        return $this->belongsTo(Sede::class);
    }

    /**
     * Votos recibidos por el candidato
     */
    public function votos(): HasMany
    {
        return $this->hasMany(Voto::class);
    }

    /**
     * Total de votos del candidato
     */
    public function getTotalVotosAttribute(): int
    {
        return $this->votos()->count();
    }

    /**
     * Porcentaje de votos respecto al total de su tipo en su sede
     */
    public function getPorcentajeVotosAttribute(): float
    {
        $totalVotosTipo = Voto::whereHas('candidato', function ($query) {
            $query->where('sede_id', $this->sede_id)
                  ->where('tipo', $this->tipo);
        })->count();

        if ($totalVotosTipo === 0) {
            return 0;
        }

        return round(($this->total_votos / $totalVotosTipo) * 100, 2);
    }
}
