<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;

class User extends Authenticatable
{
    /** @use HasFactory<\Database\Factories\UserFactory> */
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'documento',
        'grado_id',
        'role',
        'habilitado',
        'ha_votado_personero',
        'ha_votado_contralor',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password' => 'hashed',
            'habilitado' => 'boolean',
            'ha_votado_personero' => 'boolean',
            'ha_votado_contralor' => 'boolean',
        ];
    }

    /**
     * Grado del estudiante
     */
    public function grado(): BelongsTo
    {
        return $this->belongsTo(Grado::class);
    }

    /**
     * Votos emitidos por el usuario
     */
    public function votos(): HasMany
    {
        return $this->hasMany(Voto::class);
    }

    /**
     * Verificar si el usuario es administrador
     */
    public function isAdmin(): bool
    {
        return $this->role === 'admin';
    }

    /**
     * Verificar si el estudiante puede votar
     */
    public function puedeVotar(): bool
    {
        return $this->habilitado && $this->role === 'estudiante';
    }

    /**
     * Verificar si ya votó por personero
     */
    public function yaVotoPersonero(): bool
    {
        return $this->ha_votado_personero;
    }

    /**
     * Verificar si ya votó por contralor
     */
    public function yaVotoContralor(): bool
    {
        return $this->ha_votado_contralor;
    }

    /**
     * Obtener la sede del estudiante a través del grado
     */
    public function getSede()
    {
        return $this->grado?->sede;
    }
}
