'use client';

import { useEffect, useState } from 'react';
import { adminApi } from '@/lib/api';

interface Sede {
    id: number;
    nombre: string;
}

interface Candidato {
    id: number;
    nombre: string;
    tipo: 'personero' | 'contralor';
    propuestas: string;
    foto?: string;
    activo: boolean;
    sede_id: number;
    sede?: Sede;
    votos_count?: number;
}

export default function CandidatosPage() {
    const [candidatos, setCandidatos] = useState<Candidato[]>([]);
    const [sedes, setSedes] = useState<Sede[]>([]);
    const [loading, setLoading] = useState(true);
    const [filterSedeId, setFilterSedeId] = useState<number | undefined>();
    const [filterTipo, setFilterTipo] = useState<string>('');
    const [showModal, setShowModal] = useState(false);
    const [editingCandidato, setEditingCandidato] = useState<Candidato | null>(null);
    const [formData, setFormData] = useState({
        nombre: '',
        tipo: 'personero' as 'personero' | 'contralor',
        propuestas: '',
        sede_id: 0,
        activo: true,
    });
    const [fotoFile, setFotoFile] = useState<File | null>(null);
    const [fotoPreview, setFotoPreview] = useState<string>('');
    const [saving, setSaving] = useState(false);
    const [error, setError] = useState('');

    const fetchSedes = async () => {
        try {
            const response = await adminApi.getSedes();
            setSedes(response.data);
        } catch (error) {
            console.error('Error fetching sedes:', error);
        }
    };

    const fetchCandidatos = async () => {
        setLoading(true);
        try {
            const filters: { sede_id?: number; tipo?: string } = {};
            if (filterSedeId) filters.sede_id = filterSedeId;
            if (filterTipo) filters.tipo = filterTipo;
            const response = await adminApi.getCandidatos(filters);
            setCandidatos(response.data);
        } catch (error) {
            console.error('Error fetching candidatos:', error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchSedes();
    }, []);

    useEffect(() => {
        fetchCandidatos();
    }, [filterSedeId, filterTipo]);

    const handleFotoChange = (e: React.ChangeEvent<HTMLInputElement>) => {
        const file = e.target.files?.[0];
        if (file) {
            setFotoFile(file);
            const reader = new FileReader();
            reader.onloadend = () => {
                setFotoPreview(reader.result as string);
            };
            reader.readAsDataURL(file);
        }
    };

    const openCreateModal = () => {
        setEditingCandidato(null);
        setFormData({
            nombre: '',
            tipo: 'personero',
            propuestas: '',
            sede_id: sedes[0]?.id || 0,
            activo: true,
        });
        setFotoFile(null);
        setFotoPreview('');
        setError('');
        setShowModal(true);
    };

    const openEditModal = (candidato: Candidato) => {
        setEditingCandidato(candidato);
        setFormData({
            nombre: candidato.nombre,
            tipo: candidato.tipo,
            propuestas: candidato.propuestas,
            sede_id: candidato.sede_id,
            activo: candidato.activo,
        });
        setFotoFile(null);
        setFotoPreview(candidato.foto ? `${process.env.NEXT_PUBLIC_API_URL?.replace('/api', '')}/storage/${candidato.foto}` : '');
        setError('');
        setShowModal(true);
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!formData.nombre.trim() || !formData.sede_id) {
            setError('Nombre y sede son requeridos');
            return;
        }

        setSaving(true);
        setError('');

        try {
            const data = new FormData();
            data.append('nombre', formData.nombre);
            data.append('tipo', formData.tipo);
            data.append('propuestas', formData.propuestas);
            data.append('sede_id', formData.sede_id.toString());
            data.append('activo', formData.activo ? '1' : '0');
            if (fotoFile) {
                data.append('foto', fotoFile);
            }

            if (editingCandidato) {
                data.append('_method', 'PUT');
                await adminApi.updateCandidato(editingCandidato.id, data);
            } else {
                await adminApi.createCandidato(data);
            }
            setShowModal(false);
            fetchCandidatos();
        } catch (err: unknown) {
            const error = err as { response?: { data?: { message?: string } } };
            setError(error.response?.data?.message || 'Error al guardar');
        } finally {
            setSaving(false);
        }
    };

    const handleDelete = async (candidato: Candidato) => {
        if (!confirm(`¿Eliminar candidato "${candidato.nombre}"? Esta acción no se puede deshacer.`)) {
            return;
        }

        try {
            await adminApi.deleteCandidato(candidato.id);
            fetchCandidatos();
        } catch (err: unknown) {
            const error = err as { response?: { data?: { message?: string } } };
            alert(error.response?.data?.message || 'Error al eliminar');
        }
    };

    if (loading && candidatos.length === 0) {
        return (
            <div className="flex items-center justify-center h-64">
                <div className="spinner"></div>
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header con estadísticas */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
                <div className="md:col-span-2 bg-gradient-to-br from-green-500 to-green-600 rounded-2xl p-6 text-white shadow-lg">
                    <div className="flex items-center justify-between">
                        <div>
                            <p className="text-green-100 text-sm font-medium">Total Candidatos</p>
                            <h2 className="text-4xl font-bold mt-1">{candidatos.length}</h2>
                            <p className="text-green-200 text-xs mt-2">Registrados en el sistema</p>
                        </div>
                        <div className="text-6xl opacity-30">👥</div>
                    </div>
                </div>
                <div className="bg-white rounded-2xl p-6 shadow-md border border-gray-100">
                    <div className="flex items-center gap-4">
                        <div className="w-12 h-12 rounded-xl bg-green-100 flex items-center justify-center">
                            <span className="text-2xl">🎓</span>
                        </div>
                        <div>
                            <p className="text-gray-500 text-xs font-medium">Personeros</p>
                            <h3 className="text-2xl font-bold text-gray-800">
                                {candidatos.filter(c => c.tipo === 'personero').length}
                            </h3>
                        </div>
                    </div>
                </div>
                <div className="bg-white rounded-2xl p-6 shadow-md border border-gray-100">
                    <div className="flex items-center gap-4">
                        <div className="w-12 h-12 rounded-xl bg-yellow-100 flex items-center justify-center">
                            <span className="text-2xl">📋</span>
                        </div>
                        <div>
                            <p className="text-gray-500 text-xs font-medium">Contralores</p>
                            <h3 className="text-2xl font-bold text-gray-800">
                                {candidatos.filter(c => c.tipo === 'contralor').length}
                            </h3>
                        </div>
                    </div>
                </div>
            </div>

            {/* Filtros y acciones */}
            <div className="bg-white rounded-2xl p-5 shadow-md border border-gray-100">
                <div className="flex flex-wrap items-center justify-between gap-4">
                    <div className="flex items-center gap-2">
                        <span className="text-gray-400">🔍</span>
                        <span className="text-gray-600 font-medium text-sm">Filtrar por:</span>
                    </div>
                    <div className="flex flex-wrap items-center gap-3">
                        <div className="relative">
                            <select
                                value={filterSedeId || ''}
                                onChange={(e) => setFilterSedeId(e.target.value ? Number(e.target.value) : undefined)}
                                className="appearance-none bg-gray-50 border border-gray-200 rounded-xl px-4 py-2.5 pr-10 text-sm font-medium text-gray-700 focus:ring-2 focus:ring-green-500 focus:border-green-500 cursor-pointer hover:bg-gray-100 transition"
                            >
                                <option value="">🏛️ Todas las Sedes</option>
                                {sedes.map(sede => (
                                    <option key={sede.id} value={sede.id}>{sede.nombre}</option>
                                ))}
                            </select>
                            <span className="absolute right-3 top-1/2 -translate-y-1/2 pointer-events-none text-gray-400">▼</span>
                        </div>
                        <div className="relative">
                            <select
                                value={filterTipo}
                                onChange={(e) => setFilterTipo(e.target.value)}
                                className="appearance-none bg-gray-50 border border-gray-200 rounded-xl px-4 py-2.5 pr-10 text-sm font-medium text-gray-700 focus:ring-2 focus:ring-green-500 focus:border-green-500 cursor-pointer hover:bg-gray-100 transition"
                            >
                                <option value="">📊 Todos los Tipos</option>
                                <option value="personero">🎓 Personero</option>
                                <option value="contralor">📋 Contralor</option>
                            </select>
                            <span className="absolute right-3 top-1/2 -translate-y-1/2 pointer-events-none text-gray-400">▼</span>
                        </div>
                        <button
                            onClick={openCreateModal}
                            className="bg-gradient-to-r from-green-500 to-green-600 text-white px-5 py-2.5 rounded-xl font-medium text-sm shadow-md hover:shadow-lg hover:from-green-600 hover:to-green-700 transition-all duration-200 flex items-center gap-2"
                        >
                            <span className="text-lg">➕</span>
                            Nuevo Candidato
                        </button>
                    </div>
                </div>
            </div>

            {/* Grid de Candidatos */}
            <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-6">
                {candidatos.length > 0 ? (
                    candidatos.map(candidato => (
                        <div key={candidato.id} className="card relative">
                            {/* Badge de tipo */}
                            <div className="absolute top-4 right-4">
                                <span className={`badge ${candidato.tipo === 'personero' ? 'badge-green' : 'badge-yellow'}`}>
                                    {candidato.tipo === 'personero' ? '🎓 Personero' : '📋 Contralor'}
                                </span>
                            </div>

                            {/* Foto */}
                            <div className="flex justify-center mb-4">
                                {candidato.foto ? (
                                    <img
                                        src={`${process.env.NEXT_PUBLIC_API_URL?.replace('/api', '')}/storage/${candidato.foto}`}
                                        alt={candidato.nombre}
                                        className="w-24 h-24 rounded-full object-cover border-4 border-green-100"
                                    />
                                ) : (
                                    <div className="w-24 h-24 rounded-full bg-gradient-to-br from-green-400 to-green-600 flex items-center justify-center text-white text-2xl font-bold border-4 border-green-100">
                                        {candidato.nombre.substring(0, 2).toUpperCase()}
                                    </div>
                                )}
                            </div>

                            {/* Info */}
                            <div className="text-center mb-4">
                                <h3 className="font-bold text-lg text-gray-800">{candidato.nombre}</h3>
                                <p className="text-sm text-gray-500">{candidato.sede?.nombre}</p>
                                <span className={`badge mt-2 ${candidato.activo ? 'badge-green' : 'badge-red'}`}>
                                    {candidato.activo ? 'Activo' : 'Inactivo'}
                                </span>
                            </div>

                            {/* Propuestas */}
                            <div className="text-sm text-gray-600 mb-4 line-clamp-3">
                                {candidato.propuestas || 'Sin propuestas registradas'}
                            </div>

                            {/* Votos */}
                            <div className="text-center py-2 bg-green-50 rounded-lg mb-4">
                                <span className="text-2xl font-bold text-green-600">{candidato.votos_count || 0}</span>
                                <span className="text-sm text-gray-500 ml-2">votos</span>
                            </div>

                            {/* Acciones */}
                            <div className="flex gap-2">
                                <button
                                    onClick={() => openEditModal(candidato)}
                                    className="flex-1 py-2 text-blue-600 bg-blue-50 rounded-lg hover:bg-blue-100 transition text-sm font-medium"
                                >
                                    ✏️ Editar
                                </button>
                                <button
                                    onClick={() => handleDelete(candidato)}
                                    className="flex-1 py-2 text-red-600 bg-red-50 rounded-lg hover:bg-red-100 transition text-sm font-medium"
                                >
                                    🗑️ Eliminar
                                </button>
                            </div>
                        </div>
                    ))
                ) : (
                    <div className="col-span-full text-center py-12 text-gray-400">
                        No hay candidatos registrados
                    </div>
                )}
            </div>

            {/* Modal */}
            {showModal && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
                    <div className="bg-white rounded-2xl p-6 w-full max-w-lg shadow-xl max-h-[90vh] overflow-y-auto">
                        <h2 className="text-xl font-bold mb-4">
                            {editingCandidato ? 'Editar Candidato' : 'Nuevo Candidato'}
                        </h2>

                        {error && <div className="alert alert-error mb-4">{error}</div>}

                        <form onSubmit={handleSubmit} className="space-y-4">
                            {/* Foto */}
                            <div className="text-center">
                                <div className="mb-2">
                                    {fotoPreview ? (
                                        <img
                                            src={fotoPreview}
                                            alt="Preview"
                                            className="w-24 h-24 mx-auto rounded-full object-cover border-4 border-green-100"
                                        />
                                    ) : (
                                        <div className="w-24 h-24 mx-auto rounded-full bg-gray-200 flex items-center justify-center text-gray-400">
                                            📷
                                        </div>
                                    )}
                                </div>
                                <label className="btn-secondary text-sm cursor-pointer inline-block">
                                    Subir Foto
                                    <input
                                        type="file"
                                        accept="image/*"
                                        onChange={handleFotoChange}
                                        className="hidden"
                                    />
                                </label>
                            </div>

                            <div>
                                <label className="form-label">Nombre</label>
                                <input
                                    type="text"
                                    value={formData.nombre}
                                    onChange={(e) => setFormData(prev => ({ ...prev, nombre: e.target.value }))}
                                    className="form-input"
                                    placeholder="Nombre del candidato"
                                    required
                                />
                            </div>

                            <div className="grid grid-cols-2 gap-4">
                                <div>
                                    <label className="form-label">Tipo</label>
                                    <select
                                        value={formData.tipo}
                                        onChange={(e) => setFormData(prev => ({ ...prev, tipo: e.target.value as 'personero' | 'contralor' }))}
                                        className="form-input"
                                        required
                                    >
                                        <option value="personero">Personero</option>
                                        <option value="contralor">Contralor</option>
                                    </select>
                                </div>
                                <div>
                                    <label className="form-label">Sede</label>
                                    <select
                                        value={formData.sede_id}
                                        onChange={(e) => setFormData(prev => ({ ...prev, sede_id: Number(e.target.value) }))}
                                        className="form-input"
                                        required
                                    >
                                        <option value="">Seleccionar</option>
                                        {sedes.map(sede => (
                                            <option key={sede.id} value={sede.id}>{sede.nombre}</option>
                                        ))}
                                    </select>
                                </div>
                            </div>

                            <div>
                                <label className="form-label">Propuestas</label>
                                <textarea
                                    value={formData.propuestas}
                                    onChange={(e) => setFormData(prev => ({ ...prev, propuestas: e.target.value }))}
                                    className="form-input min-h-[100px]"
                                    placeholder="Propuestas del candidato"
                                />
                            </div>

                            <div className="flex items-center gap-3">
                                <input
                                    type="checkbox"
                                    id="activo"
                                    checked={formData.activo}
                                    onChange={(e) => setFormData(prev => ({ ...prev, activo: e.target.checked }))}
                                    className="w-5 h-5 rounded border-gray-300 text-green-600 focus:ring-green-500"
                                />
                                <label htmlFor="activo" className="text-sm font-medium text-gray-700">
                                    Candidato activo
                                </label>
                            </div>

                            <div className="flex gap-3 pt-4">
                                <button
                                    type="button"
                                    onClick={() => setShowModal(false)}
                                    className="flex-1 px-4 py-3 border border-gray-300 rounded-xl hover:bg-gray-50 transition"
                                >
                                    Cancelar
                                </button>
                                <button
                                    type="submit"
                                    disabled={saving}
                                    className="flex-1 btn-primary disabled:opacity-50"
                                >
                                    {saving ? 'Guardando...' : 'Guardar'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
