'use client';

import { useEffect, useState } from 'react';
import { adminApi } from '@/lib/api';

interface Sede {
    id: number;
    nombre: string;
}

interface Grado {
    id: number;
    nombre: string;
    sede_id: number;
}

interface Estudiante {
    id: number;
    name: string;
    email: string;
    documento: string;
    grado_id: number;
    grado?: {
        id: number;
        nombre: string;
        sede?: Sede;
    };
    habilitado: boolean;
    ha_votado_personero: boolean;
    ha_votado_contralor: boolean;
}

export default function EstudiantesPage() {
    const [estudiantes, setEstudiantes] = useState<Estudiante[]>([]);
    const [sedes, setSedes] = useState<Sede[]>([]);
    const [grados, setGrados] = useState<Grado[]>([]);
    const [loading, setLoading] = useState(true);
    const [filterSedeId, setFilterSedeId] = useState<number | undefined>();
    const [filterGradoId, setFilterGradoId] = useState<number | undefined>();
    const [filterHabilitado, setFilterHabilitado] = useState<string>('');
    const [showModal, setShowModal] = useState(false);
    const [editingEstudiante, setEditingEstudiante] = useState<Estudiante | null>(null);
    const [formData, setFormData] = useState({
        name: '',
        email: '',
        documento: '',
        grado_id: 0,
        habilitado: true,
    });
    const [saving, setSaving] = useState(false);
    const [error, setError] = useState('');
    const [actionLoading, setActionLoading] = useState('');
    const [showImportModal, setShowImportModal] = useState(false);
    const [importFile, setImportFile] = useState<File | null>(null);
    const [importGradoId, setImportGradoId] = useState<number>(0);
    const [importResult, setImportResult] = useState<{ imported: number; skipped: number; errors: string[]; total_errors: number } | null>(null);

    const fetchSedes = async () => {
        try {
            const response = await adminApi.getSedes();
            setSedes(response.data);
        } catch (error) {
            console.error('Error fetching sedes:', error);
        }
    };

    const fetchGrados = async () => {
        try {
            const response = await adminApi.getGrados();
            setGrados(response.data);
        } catch (error) {
            console.error('Error fetching grados:', error);
        }
    };

    const fetchEstudiantes = async () => {
        setLoading(true);
        try {
            const filters: { grado_id?: number; sede_id?: number; habilitado?: boolean } = {};
            if (filterGradoId) filters.grado_id = filterGradoId;
            if (filterSedeId) filters.sede_id = filterSedeId;
            if (filterHabilitado !== '') filters.habilitado = filterHabilitado === 'true';
            const response = await adminApi.getEstudiantes(filters);
            setEstudiantes(response.data);
        } catch (error) {
            console.error('Error fetching estudiantes:', error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchSedes();
        fetchGrados();
    }, []);

    useEffect(() => {
        fetchEstudiantes();
    }, [filterSedeId, filterGradoId, filterHabilitado]);

    const filteredGrados = filterSedeId
        ? grados.filter(g => g.sede_id === filterSedeId)
        : grados;

    const openCreateModal = () => {
        setEditingEstudiante(null);
        setFormData({
            name: '',
            email: '',
            documento: '',
            grado_id: grados[0]?.id || 0,
            habilitado: true,
        });
        setError('');
        setShowModal(true);
    };

    const openEditModal = (estudiante: Estudiante) => {
        setEditingEstudiante(estudiante);
        setFormData({
            name: estudiante.name,
            email: estudiante.email,
            documento: estudiante.documento,
            grado_id: estudiante.grado_id,
            habilitado: estudiante.habilitado,
        });
        setError('');
        setShowModal(true);
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!formData.name.trim() || !formData.documento.trim() || !formData.grado_id) {
            setError('Nombre, documento y grado son requeridos');
            return;
        }

        setSaving(true);
        setError('');

        try {
            if (editingEstudiante) {
                await adminApi.updateEstudiante(editingEstudiante.id, formData);
            } else {
                await adminApi.createEstudiante(formData);
            }
            setShowModal(false);
            fetchEstudiantes();
        } catch (err: unknown) {
            const error = err as { response?: { data?: { message?: string } } };
            setError(error.response?.data?.message || 'Error al guardar');
        } finally {
            setSaving(false);
        }
    };

    const handleDelete = async (estudiante: Estudiante) => {
        if (!confirm(`¿Eliminar estudiante "${estudiante.name}"? Esta acción no se puede deshacer.`)) {
            return;
        }

        try {
            await adminApi.deleteEstudiante(estudiante.id);
            fetchEstudiantes();
        } catch (err: unknown) {
            const error = err as { response?: { data?: { message?: string } } };
            alert(error.response?.data?.message || 'Error al eliminar');
        }
    };

    const handleHabilitarTodos = async () => {
        if (!confirm('¿Habilitar todos los estudiantes para votar?')) return;
        setActionLoading('habilitar');
        try {
            await adminApi.habilitarTodos();
            fetchEstudiantes();
        } catch (error) {
            console.error('Error:', error);
            alert('Error al habilitar estudiantes');
        } finally {
            setActionLoading('');
        }
    };

    const handleResetearVotos = async () => {
        if (!confirm('⚠️ ¿Resetear TODOS los votos? Esta acción eliminará todos los votos registrados y no se puede deshacer.')) return;
        setActionLoading('resetear');
        try {
            await adminApi.resetearVotos();
            fetchEstudiantes();
        } catch (error) {
            console.error('Error:', error);
            alert('Error al resetear votos');
        } finally {
            setActionLoading('');
        }
    };

    const handleExport = async () => {
        setActionLoading('export');
        try {
            const response = await adminApi.exportEstudiantes({
                sede_id: filterSedeId,
                grado_id: filterGradoId,
            });
            const blob = new Blob([response.data], { type: 'text/csv' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = `estudiantes_${new Date().toISOString().split('T')[0]}.csv`;
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            window.URL.revokeObjectURL(url);
        } catch (error) {
            console.error('Error:', error);
            alert('Error al exportar estudiantes');
        } finally {
            setActionLoading('');
        }
    };

    const handleImport = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!importFile || !importGradoId) {
            setError('Selecciona un archivo y un grado');
            return;
        }

        setActionLoading('import');
        setError('');
        setImportResult(null);

        try {
            const formData = new FormData();
            formData.append('file', importFile);
            formData.append('grado_id', String(importGradoId));

            const response = await adminApi.importEstudiantes(formData);
            setImportResult(response.data);
            fetchEstudiantes();
        } catch (err: unknown) {
            const error = err as { response?: { data?: { message?: string } } };
            setError(error.response?.data?.message || 'Error al importar');
        } finally {
            setActionLoading('');
        }
    };

    if (loading && estudiantes.length === 0) {
        return (
            <div className="flex items-center justify-center h-64">
                <div className="spinner"></div>
            </div>
        );
    }

    return (
        <div className="space-y-6">
            <div className="flex flex-wrap justify-between items-start gap-4">
                <div>
                    <p className="text-gray-500">Administra los estudiantes habilitados para votar</p>
                </div>
                <div className="flex flex-wrap gap-2">
                    <button
                        onClick={handleExport}
                        disabled={!!actionLoading}
                        className="px-4 py-2 bg-purple-500 text-white rounded-lg hover:bg-purple-600 transition text-sm font-medium disabled:opacity-50"
                    >
                        {actionLoading === 'export' ? '...' : '📥 Exportar CSV'}
                    </button>
                    <button
                        onClick={() => {
                            setShowImportModal(true);
                            setImportFile(null);
                            setImportGradoId(grados[0]?.id || 0);
                            setImportResult(null);
                            setError('');
                        }}
                        disabled={!!actionLoading}
                        className="px-4 py-2 bg-indigo-500 text-white rounded-lg hover:bg-indigo-600 transition text-sm font-medium disabled:opacity-50"
                    >
                        📤 Importar CSV
                    </button>
                    <button
                        onClick={handleHabilitarTodos}
                        disabled={!!actionLoading}
                        className="px-4 py-2 bg-blue-500 text-white rounded-lg hover:bg-blue-600 transition text-sm font-medium disabled:opacity-50"
                    >
                        {actionLoading === 'habilitar' ? '...' : '✅ Habilitar Todos'}
                    </button>
                    <button
                        onClick={handleResetearVotos}
                        disabled={!!actionLoading}
                        className="px-4 py-2 bg-red-500 text-white rounded-lg hover:bg-red-600 transition text-sm font-medium disabled:opacity-50"
                    >
                        {actionLoading === 'resetear' ? '...' : '🔄 Resetear Votos'}
                    </button>
                    <button onClick={openCreateModal} className="btn-primary flex items-center gap-2">
                        <span>➕</span> Nuevo Estudiante
                    </button>
                </div>
            </div>

            {/* Filtros */}
            <div className="card">
                <div className="flex flex-wrap gap-4">
                    <div className="flex-1 min-w-[200px]">
                        <label className="form-label">Sede</label>
                        <select
                            value={filterSedeId || ''}
                            onChange={(e) => {
                                setFilterSedeId(e.target.value ? Number(e.target.value) : undefined);
                                setFilterGradoId(undefined);
                            }}
                            className="form-input"
                        >
                            <option value="">Todas las Sedes</option>
                            {sedes.map(sede => (
                                <option key={sede.id} value={sede.id}>{sede.nombre}</option>
                            ))}
                        </select>
                    </div>
                    <div className="flex-1 min-w-[200px]">
                        <label className="form-label">Grado</label>
                        <select
                            value={filterGradoId || ''}
                            onChange={(e) => setFilterGradoId(e.target.value ? Number(e.target.value) : undefined)}
                            className="form-input"
                        >
                            <option value="">Todos los Grados</option>
                            {filteredGrados.map(grado => (
                                <option key={grado.id} value={grado.id}>{grado.nombre}</option>
                            ))}
                        </select>
                    </div>
                    <div className="flex-1 min-w-[200px]">
                        <label className="form-label">Estado</label>
                        <select
                            value={filterHabilitado}
                            onChange={(e) => setFilterHabilitado(e.target.value)}
                            className="form-input"
                        >
                            <option value="">Todos</option>
                            <option value="true">Habilitados</option>
                            <option value="false">No Habilitados</option>
                        </select>
                    </div>
                </div>
            </div>

            {/* Estadísticas rápidas */}
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <div className="card text-center">
                    <div className="text-2xl font-bold text-gray-800">{estudiantes.length}</div>
                    <div className="text-sm text-gray-500">Total</div>
                </div>
                <div className="card text-center">
                    <div className="text-2xl font-bold text-green-600">
                        {estudiantes.filter(e => e.habilitado).length}
                    </div>
                    <div className="text-sm text-gray-500">Habilitados</div>
                </div>
                <div className="card text-center">
                    <div className="text-2xl font-bold text-blue-600">
                        {estudiantes.filter(e => e.ha_votado_personero || e.ha_votado_contralor).length}
                    </div>
                    <div className="text-sm text-gray-500">Han Votado</div>
                </div>
                <div className="card text-center">
                    <div className="text-2xl font-bold text-yellow-600">
                        {estudiantes.filter(e => e.habilitado && !e.ha_votado_personero && !e.ha_votado_contralor).length}
                    </div>
                    <div className="text-sm text-gray-500">Pendientes</div>
                </div>
            </div>

            {/* Tabla */}
            <div className="card overflow-x-auto">
                <table className="w-full min-w-[700px]">
                    <thead className="bg-gray-50">
                        <tr>
                            <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Nombre</th>
                            <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Documento</th>
                            <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Grado</th>
                            <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Estado</th>
                            <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Votó</th>
                            <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Acciones</th>
                        </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-100">
                        {estudiantes.length > 0 ? (
                            estudiantes.map(estudiante => (
                                <tr key={estudiante.id} className="hover:bg-gray-50">
                                    <td className="px-4 py-3">
                                        <div>
                                            <div className="font-medium">{estudiante.name}</div>
                                            <div className="text-xs text-gray-500">{estudiante.email}</div>
                                        </div>
                                    </td>
                                    <td className="px-4 py-3 font-mono text-sm">{estudiante.documento}</td>
                                    <td className="px-4 py-3 text-sm">
                                        <div>{estudiante.grado?.nombre}</div>
                                        <div className="text-xs text-gray-500">{estudiante.grado?.sede?.nombre}</div>
                                    </td>
                                    <td className="px-4 py-3 text-center">
                                        <span className={`badge ${estudiante.habilitado ? 'badge-green' : 'badge-red'}`}>
                                            {estudiante.habilitado ? 'Habilitado' : 'No Habilitado'}
                                        </span>
                                    </td>
                                    <td className="px-4 py-3 text-center">
                                        <div className="flex justify-center gap-1">
                                            <span className={`badge ${estudiante.ha_votado_personero ? 'badge-green' : 'badge-yellow'}`} title="Personero">
                                                🎓 {estudiante.ha_votado_personero ? '✓' : '○'}
                                            </span>
                                            <span className={`badge ${estudiante.ha_votado_contralor ? 'badge-green' : 'badge-yellow'}`} title="Contralor">
                                                📋 {estudiante.ha_votado_contralor ? '✓' : '○'}
                                            </span>
                                        </div>
                                    </td>
                                    <td className="px-4 py-3">
                                        <div className="flex items-center justify-center gap-2">
                                            <button
                                                onClick={() => openEditModal(estudiante)}
                                                className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition"
                                                title="Editar"
                                            >
                                                ✏️
                                            </button>
                                            <button
                                                onClick={() => handleDelete(estudiante)}
                                                className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition"
                                                title="Eliminar"
                                            >
                                                🗑️
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            ))
                        ) : (
                            <tr>
                                <td colSpan={6} className="px-4 py-8 text-center text-gray-400">
                                    No hay estudiantes registrados
                                </td>
                            </tr>
                        )}
                    </tbody>
                </table>
            </div>

            {/* Modal */}
            {showModal && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
                    <div className="bg-white rounded-2xl p-6 w-full max-w-lg shadow-xl max-h-[90vh] overflow-y-auto">
                        <h2 className="text-xl font-bold mb-4">
                            {editingEstudiante ? 'Editar Estudiante' : 'Nuevo Estudiante'}
                        </h2>

                        {error && <div className="alert alert-error mb-4">{error}</div>}

                        <form onSubmit={handleSubmit} className="space-y-4">
                            <div>
                                <label className="form-label">Nombre Completo</label>
                                <input
                                    type="text"
                                    value={formData.name}
                                    onChange={(e) => setFormData(prev => ({ ...prev, name: e.target.value }))}
                                    className="form-input"
                                    placeholder="Nombre del estudiante"
                                    required
                                />
                            </div>

                            <div>
                                <label className="form-label">Número de Documento</label>
                                <input
                                    type="text"
                                    value={formData.documento}
                                    onChange={(e) => setFormData(prev => ({ ...prev, documento: e.target.value }))}
                                    className="form-input"
                                    placeholder="1234567890"
                                    required
                                />
                            </div>

                            <div>
                                <label className="form-label">Grado</label>
                                <select
                                    value={formData.grado_id}
                                    onChange={(e) => setFormData(prev => ({ ...prev, grado_id: Number(e.target.value) }))}
                                    className="form-input"
                                    required
                                >
                                    <option value="">Seleccionar grado</option>
                                    {grados.map(grado => (
                                        <option key={grado.id} value={grado.id}>{grado.nombre}</option>
                                    ))}
                                </select>
                            </div>

                            <div className="flex items-center gap-3">
                                <input
                                    type="checkbox"
                                    id="habilitado"
                                    checked={formData.habilitado}
                                    onChange={(e) => setFormData(prev => ({ ...prev, habilitado: e.target.checked }))}
                                    className="w-5 h-5 rounded border-gray-300 text-green-600 focus:ring-green-500"
                                />
                                <label htmlFor="habilitado" className="text-sm font-medium text-gray-700">
                                    Habilitado para votar
                                </label>
                            </div>

                            <div className="flex gap-3 pt-4">
                                <button
                                    type="button"
                                    onClick={() => setShowModal(false)}
                                    className="flex-1 px-4 py-3 border border-gray-300 rounded-xl hover:bg-gray-50 transition"
                                >
                                    Cancelar
                                </button>
                                <button
                                    type="submit"
                                    disabled={saving}
                                    className="flex-1 btn-primary disabled:opacity-50"
                                >
                                    {saving ? 'Guardando...' : 'Guardar'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}

            {/* Import Modal */}
            {showImportModal && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50 p-4">
                    <div className="bg-white rounded-2xl p-6 w-full max-w-lg shadow-xl max-h-[90vh] overflow-y-auto">
                        <h2 className="text-xl font-bold mb-4">📤 Importar Estudiantes</h2>

                        {error && <div className="alert alert-error mb-4">{error}</div>}

                        {importResult && (
                            <div className="mb-4 p-4 bg-green-50 border border-green-200 rounded-lg">
                                <div className="text-green-800 font-semibold mb-2">✅ Importación completada</div>
                                <div className="text-sm space-y-1">
                                    <div>✓ Importados: <strong>{importResult.imported}</strong></div>
                                    <div>⏭️ Omitidos (duplicados): <strong>{importResult.skipped}</strong></div>
                                    {importResult.total_errors > 0 && (
                                        <div className="text-red-600">
                                            ❌ Errores: <strong>{importResult.total_errors}</strong>
                                            <ul className="mt-1 text-xs list-disc list-inside">
                                                {importResult.errors.map((err, i) => (
                                                    <li key={i}>{err}</li>
                                                ))}
                                            </ul>
                                        </div>
                                    )}
                                </div>
                            </div>
                        )}

                        <form onSubmit={handleImport} className="space-y-4">
                            <div className="p-4 bg-gray-50 rounded-lg text-sm text-gray-600">
                                <strong>Formato del archivo CSV:</strong>
                                <pre className="mt-2 text-xs bg-white p-2 rounded border overflow-x-auto">
                                    documento,nombre{'\n'}
                                    1234567890,Juan Pérez{'\n'}
                                    0987654321,María García
                                </pre>
                                <p className="mt-2 text-xs">La primera fila (encabezado) será ignorada.</p>
                            </div>

                            <div>
                                <label className="form-label">Archivo CSV</label>
                                <input
                                    type="file"
                                    accept=".csv,.txt"
                                    onChange={(e) => setImportFile(e.target.files?.[0] || null)}
                                    className="form-input"
                                    required
                                />
                            </div>

                            <div>
                                <label className="form-label">Grado destino (para todos los estudiantes)</label>
                                <select
                                    value={importGradoId}
                                    onChange={(e) => setImportGradoId(Number(e.target.value))}
                                    className="form-input"
                                    required
                                >
                                    <option value="">Seleccionar grado</option>
                                    {grados.map(grado => (
                                        <option key={grado.id} value={grado.id}>{grado.nombre}</option>
                                    ))}
                                </select>
                            </div>

                            <div className="flex gap-3 pt-4">
                                <button
                                    type="button"
                                    onClick={() => setShowImportModal(false)}
                                    className="flex-1 px-4 py-3 border border-gray-300 rounded-xl hover:bg-gray-50 transition"
                                >
                                    Cerrar
                                </button>
                                <button
                                    type="submit"
                                    disabled={actionLoading === 'import'}
                                    className="flex-1 btn-primary disabled:opacity-50"
                                >
                                    {actionLoading === 'import' ? 'Importando...' : 'Importar'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
