'use client';

import { useEffect, useState } from 'react';
import { adminApi } from '@/lib/api';

interface Sede {
    id: number;
    nombre: string;
}

interface Grado {
    id: number;
    nombre: string;
    sede_id: number;
    sede?: Sede;
    estudiantes_count?: number;
}

export default function GradosPage() {
    const [grados, setGrados] = useState<Grado[]>([]);
    const [sedes, setSedes] = useState<Sede[]>([]);
    const [loading, setLoading] = useState(true);
    const [filterSedeId, setFilterSedeId] = useState<number | undefined>();
    const [showModal, setShowModal] = useState(false);
    const [editingGrado, setEditingGrado] = useState<Grado | null>(null);
    const [formData, setFormData] = useState({ nombre: '', sede_id: 0 });
    const [saving, setSaving] = useState(false);
    const [error, setError] = useState('');

    const fetchSedes = async () => {
        try {
            const response = await adminApi.getSedes();
            setSedes(response.data);
        } catch (error) {
            console.error('Error fetching sedes:', error);
        }
    };

    const fetchGrados = async () => {
        setLoading(true);
        try {
            const response = await adminApi.getGrados();
            let data = response.data;
            if (filterSedeId) {
                data = data.filter((g: Grado) => g.sede_id === filterSedeId);
            }
            setGrados(data);
        } catch (error) {
            console.error('Error fetching grados:', error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchSedes();
    }, []);

    useEffect(() => {
        fetchGrados();
    }, [filterSedeId]);

    const openCreateModal = () => {
        setEditingGrado(null);
        setFormData({ nombre: '', sede_id: sedes[0]?.id || 0 });
        setError('');
        setShowModal(true);
    };

    const openEditModal = (grado: Grado) => {
        setEditingGrado(grado);
        setFormData({ nombre: grado.nombre, sede_id: grado.sede_id });
        setError('');
        setShowModal(true);
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!formData.nombre.trim() || !formData.sede_id) {
            setError('Nombre y sede son requeridos');
            return;
        }

        setSaving(true);
        setError('');

        try {
            if (editingGrado) {
                await adminApi.updateGrado(editingGrado.id, formData);
            } else {
                await adminApi.createGrado(formData);
            }
            setShowModal(false);
            fetchGrados();
        } catch (err: unknown) {
            const error = err as { response?: { data?: { message?: string } } };
            setError(error.response?.data?.message || 'Error al guardar');
        } finally {
            setSaving(false);
        }
    };

    const handleDelete = async (grado: Grado) => {
        if (!confirm(`¿Eliminar grado "${grado.nombre}"? Esta acción no se puede deshacer.`)) {
            return;
        }

        try {
            await adminApi.deleteGrado(grado.id);
            fetchGrados();
        } catch (err: unknown) {
            const error = err as { response?: { data?: { message?: string } } };
            alert(error.response?.data?.message || 'Error al eliminar');
        }
    };

    if (loading && grados.length === 0) {
        return (
            <div className="flex items-center justify-center h-64">
                <div className="spinner"></div>
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header */}
            <div className="flex flex-wrap justify-between items-center gap-4">
                <div>
                    <p className="text-gray-500">Administra los grados por sede</p>
                </div>
                <div className="flex items-center gap-4">
                    <select
                        value={filterSedeId || ''}
                        onChange={(e) => setFilterSedeId(e.target.value ? Number(e.target.value) : undefined)}
                        className="form-input"
                    >
                        <option value="">Todas las Sedes</option>
                        {sedes.map(sede => (
                            <option key={sede.id} value={sede.id}>{sede.nombre}</option>
                        ))}
                    </select>
                    <button onClick={openCreateModal} className="btn-primary flex items-center gap-2">
                        <span>➕</span> Nuevo Grado
                    </button>
                </div>
            </div>

            {/* Tabla */}
            <div className="card">
                <table className="w-full">
                    <thead className="bg-gray-50">
                        <tr>
                            <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Nombre</th>
                            <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Sede</th>
                            <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Estudiantes</th>
                            <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Acciones</th>
                        </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-100">
                        {grados.length > 0 ? (
                            grados.map(grado => (
                                <tr key={grado.id} className="hover:bg-gray-50">
                                    <td className="px-4 py-3 font-medium">{grado.nombre}</td>
                                    <td className="px-4 py-3 text-gray-500">{grado.sede?.nombre || '-'}</td>
                                    <td className="px-4 py-3 text-center">
                                        <span className="badge badge-blue">{grado.estudiantes_count || 0}</span>
                                    </td>
                                    <td className="px-4 py-3">
                                        <div className="flex items-center justify-center gap-2">
                                            <button
                                                onClick={() => openEditModal(grado)}
                                                className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition"
                                                title="Editar"
                                            >
                                                ✏️
                                            </button>
                                            <button
                                                onClick={() => handleDelete(grado)}
                                                className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition"
                                                title="Eliminar"
                                            >
                                                🗑️
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            ))
                        ) : (
                            <tr>
                                <td colSpan={4} className="px-4 py-8 text-center text-gray-400">
                                    No hay grados registrados
                                </td>
                            </tr>
                        )}
                    </tbody>
                </table>
            </div>

            {/* Modal */}
            {showModal && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50">
                    <div className="bg-white rounded-2xl p-6 w-full max-w-md shadow-xl">
                        <h2 className="text-xl font-bold mb-4">
                            {editingGrado ? 'Editar Grado' : 'Nuevo Grado'}
                        </h2>

                        {error && <div className="alert alert-error mb-4">{error}</div>}

                        <form onSubmit={handleSubmit} className="space-y-4">
                            <div>
                                <label className="form-label">Nombre del Grado</label>
                                <input
                                    type="text"
                                    value={formData.nombre}
                                    onChange={(e) => setFormData(prev => ({ ...prev, nombre: e.target.value }))}
                                    className="form-input"
                                    placeholder="Ej: Grado 6°"
                                    required
                                />
                            </div>

                            <div>
                                <label className="form-label">Sede</label>
                                <select
                                    value={formData.sede_id}
                                    onChange={(e) => setFormData(prev => ({ ...prev, sede_id: Number(e.target.value) }))}
                                    className="form-input"
                                    required
                                >
                                    <option value="">Seleccionar sede</option>
                                    {sedes.map(sede => (
                                        <option key={sede.id} value={sede.id}>{sede.nombre}</option>
                                    ))}
                                </select>
                            </div>

                            <div className="flex gap-3 pt-4">
                                <button
                                    type="button"
                                    onClick={() => setShowModal(false)}
                                    className="flex-1 px-4 py-3 border border-gray-300 rounded-xl hover:bg-gray-50 transition"
                                >
                                    Cancelar
                                </button>
                                <button
                                    type="submit"
                                    disabled={saving}
                                    className="flex-1 btn-primary disabled:opacity-50"
                                >
                                    {saving ? 'Guardando...' : 'Guardar'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
