'use client';

import { useEffect } from 'react';
import { useRouter, usePathname } from 'next/navigation';
import Link from 'next/link';
import { useAuth } from '@/lib/auth';

const navItems = [
    { href: '/admin', label: 'Dashboard', icon: '🏠', description: 'Vista general' },
    { href: '/admin/resultados', label: 'Resultados', icon: '📊', description: 'Estadísticas' },
    { href: '/admin/sedes', label: 'Sedes', icon: '🏛️', description: 'Ubicaciones' },
    { href: '/admin/grados', label: 'Grados', icon: '📚', description: 'Niveles escolares' },
    { href: '/admin/candidatos', label: 'Candidatos', icon: '👥', description: 'Personeros y contralores' },
    { href: '/admin/estudiantes', label: 'Estudiantes', icon: '🎓', description: 'Votantes' },
];

export default function AdminLayout({
    children,
}: {
    children: React.ReactNode;
}) {
    const { isAuthenticated, isAdmin, isLoading, user, logout } = useAuth();
    const router = useRouter();
    const pathname = usePathname();

    useEffect(() => {
        if (!isLoading && (!isAuthenticated || !isAdmin)) {
            router.push('/login');
        }
    }, [isAuthenticated, isAdmin, isLoading, router]);

    if (isLoading) {
        return (
            <div className="min-h-screen flex items-center justify-center bg-gradient-to-br from-green-900 to-green-700">
                <div className="spinner"></div>
            </div>
        );
    }

    if (!isAuthenticated || !isAdmin) {
        return null;
    }

    const handleLogout = async () => {
        await logout();
        router.push('/login');
    };

    return (
        <div className="flex min-h-screen bg-gray-100">
            {/* Sidebar */}
            <aside className="w-72 bg-gradient-to-b from-green-800 via-green-700 to-green-900 shadow-2xl flex flex-col fixed h-screen">
                {/* Header */}
                <div className="p-6 border-b border-white/10">
                    <div className="flex items-center gap-4">
                        <div className="w-14 h-14 rounded-xl bg-white/95 shadow-lg flex items-center justify-center transform hover:scale-105 transition-transform overflow-hidden">
                            <img
                                src={`${process.env.NEXT_PUBLIC_API_URL?.replace('/api', '') || 'http://localhost:8000'}/logo.jpg`}
                                alt="Logo"
                                className="w-full h-full object-contain p-1"
                                onError={(e) => {
                                    const target = e.target as HTMLImageElement;
                                    target.style.display = 'none';
                                    target.parentElement!.innerHTML = '<span class="text-3xl">🏫</span>';
                                }}
                            />
                        </div>
                        <div>
                            <h1 className="text-white font-bold text-lg tracking-tight">I.E. Las Huacas</h1>
                            <p className="text-green-200/70 text-xs font-medium">Panel Administrativo</p>
                        </div>
                    </div>
                </div>

                {/* Navigation */}
                <nav className="flex-1 py-6 px-4 space-y-1 overflow-y-auto">
                    {/* Principal Section */}
                    <div className="px-3 py-2 mb-2">
                        <span className="text-green-300/60 text-[10px] font-bold uppercase tracking-[0.2em]">
                            Principal
                        </span>
                    </div>

                    {navItems.slice(0, 2).map((item) => {
                        const isActive = pathname === item.href;
                        return (
                            <Link
                                key={item.href}
                                href={item.href}
                                className={`group flex items-center gap-4 px-4 py-3 rounded-xl transition-all duration-200 ${isActive
                                    ? 'bg-white/20 shadow-lg backdrop-blur-sm'
                                    : 'hover:bg-white/10'
                                    }`}
                            >
                                <span className={`text-2xl transition-transform duration-200 ${isActive ? 'scale-110' : 'group-hover:scale-110'}`}>
                                    {item.icon}
                                </span>
                                <div className="flex-1">
                                    <span className={`block font-semibold text-sm ${isActive ? 'text-white' : 'text-white/80 group-hover:text-white'}`}>
                                        {item.label}
                                    </span>
                                    <span className={`block text-[10px] ${isActive ? 'text-green-200' : 'text-white/40'}`}>
                                        {item.description}
                                    </span>
                                </div>
                                {isActive && (
                                    <div className="w-1.5 h-8 bg-white rounded-full shadow-lg"></div>
                                )}
                            </Link>
                        );
                    })}

                    {/* Gestión Section */}
                    <div className="px-3 py-2 mt-6 mb-2">
                        <span className="text-green-300/60 text-[10px] font-bold uppercase tracking-[0.2em]">
                            Gestión
                        </span>
                    </div>

                    {navItems.slice(2).map((item) => {
                        const isActive = pathname === item.href;
                        return (
                            <Link
                                key={item.href}
                                href={item.href}
                                className={`group flex items-center gap-4 px-4 py-3 rounded-xl transition-all duration-200 ${isActive
                                    ? 'bg-white/20 shadow-lg backdrop-blur-sm'
                                    : 'hover:bg-white/10'
                                    }`}
                            >
                                <span className={`text-2xl transition-transform duration-200 ${isActive ? 'scale-110' : 'group-hover:scale-110'}`}>
                                    {item.icon}
                                </span>
                                <div className="flex-1">
                                    <span className={`block font-semibold text-sm ${isActive ? 'text-white' : 'text-white/80 group-hover:text-white'}`}>
                                        {item.label}
                                    </span>
                                    <span className={`block text-[10px] ${isActive ? 'text-green-200' : 'text-white/40'}`}>
                                        {item.description}
                                    </span>
                                </div>
                                {isActive && (
                                    <div className="w-1.5 h-8 bg-white rounded-full shadow-lg"></div>
                                )}
                            </Link>
                        );
                    })}
                </nav>

                {/* Footer */}
                <div className="p-4 border-t border-white/10">
                    <div className="flex items-center gap-3 px-3 py-2 mb-3">
                        <div className="w-10 h-10 rounded-full bg-white/20 flex items-center justify-center">
                            <span className="text-lg">👤</span>
                        </div>
                        <div className="flex-1 min-w-0">
                            <p className="text-white text-sm font-medium truncate">{user?.name}</p>
                            <p className="text-green-200/60 text-xs">Administrador</p>
                        </div>
                    </div>
                    <button
                        onClick={handleLogout}
                        className="w-full px-4 py-3 rounded-xl text-white/90 bg-white/10 hover:bg-red-500/30 hover:text-red-200 transition-all duration-200 text-sm font-medium flex items-center justify-center gap-2"
                    >
                        <span>🚪</span>
                        Cerrar Sesión
                    </button>
                </div>
            </aside>

            {/* Main Content */}
            <div className="flex-1 ml-72">
                {/* Top Bar */}
                <header className="bg-white/80 backdrop-blur-md px-8 py-5 border-b border-gray-200/50 sticky top-0 z-10 shadow-sm">
                    <div className="flex justify-between items-center">
                        <div>
                            <h1 className="text-2xl font-bold text-gray-800">
                                {navItems.find((item) => item.href === pathname)?.label || 'Admin'}
                            </h1>
                            <p className="text-gray-500 text-sm">
                                {navItems.find((item) => item.href === pathname)?.description || ''}
                            </p>
                        </div>
                        <div className="flex items-center gap-4">
                            <div className="text-right">
                                <p className="text-gray-700 font-medium text-sm">{user?.name}</p>
                                <p className="text-gray-400 text-xs">Sesión activa</p>
                            </div>
                            <div className="w-10 h-10 rounded-full bg-gradient-to-br from-green-400 to-green-600 flex items-center justify-center text-white font-bold shadow-md">
                                {user?.name?.substring(0, 1).toUpperCase() || 'A'}
                            </div>
                        </div>
                    </div>
                </header>

                {/* Page Content */}
                <main className="p-8">
                    {children}
                </main>
            </div>
        </div>
    );
}
