'use client';

import { useEffect, useState } from 'react';
import { adminApi } from '@/lib/api';
import { Chart as ChartJS, ArcElement, Tooltip, Legend, CategoryScale, LinearScale, BarElement } from 'chart.js';
import { Doughnut, Bar } from 'react-chartjs-2';

ChartJS.register(ArcElement, Tooltip, Legend, CategoryScale, LinearScale, BarElement);

interface DashboardData {
    totalEstudiantes: number;
    estudiantesHabilitados: number;
    estudiantesQueVotaron: number;
    totalVotosPersonero: number;
    totalVotosContralor: number;
    votacionAbierta: boolean;
    resultadosPorSede: Array<{
        sede: { id: number; nombre: string; slug: string };
        estudiantes: number;
        votaron: number;
        porcentaje: number;
        personeros: Array<{ id: number; nombre: string; votos_count: number; foto?: string }>;
        contralores: Array<{ id: number; nombre: string; votos_count: number; foto?: string }>;
    }>;
    resultadosPorGrado: Array<{
        grado: { id: number; nombre: string };
        sede: string;
        estudiantes: number;
        votaron: number;
        porcentaje: number;
    }>;
    sedes: Array<{ id: number; nombre: string; slug: string }>;
}

export default function AdminDashboard() {
    const [data, setData] = useState<DashboardData | null>(null);
    const [loading, setLoading] = useState(true);
    const [toggling, setToggling] = useState(false);

    const fetchDashboard = async () => {
        try {
            const response = await adminApi.dashboard();
            setData(response.data);
        } catch (error) {
            console.error('Error fetching dashboard:', error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchDashboard();
    }, []);

    const handleToggleVotacion = async () => {
        if (!confirm(data?.votacionAbierta
            ? '¿Cerrar votación? Los resultados se harán públicos.'
            : '¿Abrir votación nuevamente?')) {
            return;
        }

        setToggling(true);
        try {
            await adminApi.toggleVotacion();
            fetchDashboard();
        } catch (error) {
            console.error('Error toggling votacion:', error);
        } finally {
            setToggling(false);
        }
    };

    if (loading) {
        return (
            <div className="flex items-center justify-center h-64">
                <div className="spinner"></div>
            </div>
        );
    }

    if (!data) {
        return <div className="alert alert-error">Error al cargar el dashboard</div>;
    }

    const participacionData = {
        labels: ['Han Votado', 'No Han Votado', 'No Habilitados'],
        datasets: [{
            data: [
                data.estudiantesQueVotaron,
                data.estudiantesHabilitados - data.estudiantesQueVotaron,
                data.totalEstudiantes - data.estudiantesHabilitados,
            ],
            backgroundColor: ['#22c55e', '#fbbf24', '#e5e7eb'],
            borderWidth: 0,
        }],
    };

    const votosData = {
        labels: ['Personero', 'Contralor'],
        datasets: [{
            label: 'Votos',
            data: [data.totalVotosPersonero, data.totalVotosContralor],
            backgroundColor: ['#22c55e', '#eab308'],
            borderRadius: 8,
        }],
    };

    return (
        <div className="space-y-6">
            {/* Estado de Votación */}
            <div className={`card flex flex-wrap items-center justify-between gap-4 border-2 ${data.votacionAbierta ? 'bg-green-50 border-green-200' : 'bg-red-50 border-red-200'
                }`}>
                <div className="flex items-center gap-4">
                    <div className="text-4xl">{data.votacionAbierta ? '🟢' : '🔴'}</div>
                    <div>
                        <h2 className="font-bold text-lg">
                            {data.votacionAbierta ? 'Votación Abierta' : 'Votación Cerrada'}
                        </h2>
                        <p className="text-sm text-gray-600">
                            {data.votacionAbierta ? 'Los estudiantes pueden votar' : 'Los resultados son públicos'}
                        </p>
                    </div>
                </div>
                <button
                    onClick={handleToggleVotacion}
                    disabled={toggling}
                    className={`${data.votacionAbierta
                            ? 'bg-red-500 hover:bg-red-600'
                            : 'bg-green-500 hover:bg-green-600'
                        } text-white font-bold py-3 px-6 rounded-xl transition-colors disabled:opacity-50`}
                >
                    {toggling ? 'Procesando...' : data.votacionAbierta ? '🔒 Cerrar Votación' : '🔓 Abrir Votación'}
                </button>
            </div>

            {/* Gráficas */}
            <div className="grid md:grid-cols-2 gap-6">
                <div className="card">
                    <h3 className="font-bold text-gray-700 mb-4">📊 Participación General</h3>
                    <div className="h-64">
                        <Doughnut data={participacionData} options={{ responsive: true, maintainAspectRatio: false }} />
                    </div>
                </div>
                <div className="card">
                    <h3 className="font-bold text-gray-700 mb-4">🗳️ Votos Emitidos</h3>
                    <div className="h-64">
                        <Bar
                            data={votosData}
                            options={{
                                responsive: true,
                                maintainAspectRatio: false,
                                plugins: { legend: { display: false } },
                                scales: { y: { beginAtZero: true } },
                            }}
                        />
                    </div>
                </div>
            </div>

            {/* Ganadores por Sede */}
            <div className="card">
                <h2 className="text-xl font-bold text-gray-800 mb-4">🏆 Ganadores por Sede</h2>
                <div className="grid md:grid-cols-2 gap-6">
                    {data.resultadosPorSede.map((sedeData) => (
                        <div key={sedeData.sede.id} className="p-4 bg-gradient-to-br from-green-50 to-yellow-50 rounded-2xl border border-green-100">
                            <h3 className="font-bold text-lg text-gray-800 mb-4 text-center">{sedeData.sede.nombre}</h3>
                            <div className="grid grid-cols-2 gap-4">
                                {/* Personero */}
                                <div className="text-center">
                                    <div className="text-xs text-green-600 font-semibold uppercase mb-2">🎓 Personero</div>
                                    {sedeData.personeros[0] ? (
                                        <>
                                            <div className="w-16 h-16 rounded-full mx-auto mb-2 bg-gradient-to-br from-green-400 to-green-600 flex items-center justify-center text-white text-xl font-bold border-4 border-green-400 shadow-lg">
                                                {sedeData.personeros[0].nombre.substring(0, 2).toUpperCase()}
                                            </div>
                                            <div className="font-bold text-gray-800 text-sm">{sedeData.personeros[0].nombre}</div>
                                            <div className="text-green-600 font-semibold text-sm">{sedeData.personeros[0].votos_count} votos</div>
                                        </>
                                    ) : (
                                        <div className="text-gray-400 text-sm">Sin candidatos</div>
                                    )}
                                </div>
                                {/* Contralor */}
                                <div className="text-center">
                                    <div className="text-xs text-yellow-600 font-semibold uppercase mb-2">📋 Contralor</div>
                                    {sedeData.contralores[0] ? (
                                        <>
                                            <div className="w-16 h-16 rounded-full mx-auto mb-2 bg-gradient-to-br from-yellow-400 to-yellow-600 flex items-center justify-center text-white text-xl font-bold border-4 border-yellow-400 shadow-lg">
                                                {sedeData.contralores[0].nombre.substring(0, 2).toUpperCase()}
                                            </div>
                                            <div className="font-bold text-gray-800 text-sm">{sedeData.contralores[0].nombre}</div>
                                            <div className="text-yellow-600 font-semibold text-sm">{sedeData.contralores[0].votos_count} votos</div>
                                        </>
                                    ) : (
                                        <div className="text-gray-400 text-sm">Sin candidatos</div>
                                    )}
                                </div>
                            </div>
                            {/* Participación */}
                            <div className="mt-4 pt-4 border-t border-green-200">
                                <div className="flex items-center justify-between text-sm">
                                    <span className="text-gray-500">Participación</span>
                                    <span className="font-bold text-green-600">{sedeData.porcentaje}%</span>
                                </div>
                                <div className="w-full bg-gray-200 rounded-full h-2 mt-1">
                                    <div className="bg-green-500 h-2 rounded-full" style={{ width: `${sedeData.porcentaje}%` }}></div>
                                </div>
                            </div>
                        </div>
                    ))}
                </div>
            </div>

            {/* Participación por Grado */}
            <div className="card">
                <h2 className="text-xl font-bold text-gray-800 mb-4">📚 Participación por Grado</h2>
                <div className="overflow-x-auto">
                    <table className="w-full">
                        <thead className="bg-gray-50">
                            <tr>
                                <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Grado</th>
                                <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Sede</th>
                                <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Estudiantes</th>
                                <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Votaron</th>
                                <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Participación</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-gray-100">
                            {data.resultadosPorGrado.length > 0 ? (
                                data.resultadosPorGrado.map((gradoData, index) => (
                                    <tr key={index} className="hover:bg-gray-50">
                                        <td className="px-4 py-3 font-medium">{gradoData.grado.nombre}</td>
                                        <td className="px-4 py-3 text-gray-600">{gradoData.sede}</td>
                                        <td className="px-4 py-3 text-center">{gradoData.estudiantes}</td>
                                        <td className="px-4 py-3 text-center">
                                            <span className="badge badge-blue">{gradoData.votaron}</span>
                                        </td>
                                        <td className="px-4 py-3">
                                            <div className="flex items-center gap-2">
                                                <div className="flex-1 bg-gray-200 rounded-full h-2">
                                                    <div className="bg-green-500 h-2 rounded-full" style={{ width: `${gradoData.porcentaje}%` }}></div>
                                                </div>
                                                <span className="text-sm font-medium text-gray-700 w-12">{gradoData.porcentaje}%</span>
                                            </div>
                                        </td>
                                    </tr>
                                ))
                            ) : (
                                <tr>
                                    <td colSpan={5} className="px-4 py-8 text-center text-gray-500">No hay grados con estudiantes</td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>
            </div>

            {/* Links de Votación */}
            <div className="card">
                <h2 className="text-xl font-bold text-gray-800 mb-4">🔗 Links Públicos de Votación</h2>
                <div className="grid md:grid-cols-2 lg:grid-cols-3 gap-4">
                    {data.sedes.map((sede) => (
                        <div key={sede.id} className="p-4 bg-green-50 rounded-xl">
                            <h3 className="font-semibold mb-2">{sede.nombre}</h3>
                            <a
                                href={`/sede/${sede.slug}`}
                                target="_blank"
                                rel="noopener noreferrer"
                                className="text-green-600 text-sm hover:underline break-all"
                            >
                                /sede/{sede.slug}
                            </a>
                        </div>
                    ))}
                </div>
            </div>
        </div>
    );
}
