'use client';

import { useEffect, useState } from 'react';
import { adminApi } from '@/lib/api';
import { Chart as ChartJS, ArcElement, Tooltip, Legend, CategoryScale, LinearScale, BarElement } from 'chart.js';
import { Bar } from 'react-chartjs-2';

ChartJS.register(ArcElement, Tooltip, Legend, CategoryScale, LinearScale, BarElement);

interface Sede {
    id: number;
    nombre: string;
    slug: string;
}

interface Candidato {
    id: number;
    nombre: string;
    tipo: string;
    votos_count: number;
    sede?: Sede;
}

interface ResultadosData {
    personeros: Candidato[];
    contralores: Candidato[];
    sedes: Sede[];
    chartDataPersoneros: {
        labels: string[];
        data: number[];
    };
    chartDataContralores: {
        labels: string[];
        data: number[];
    };
}

export default function ResultadosPage() {
    const [sedes, setSedes] = useState<Sede[]>([]);
    const [sedeId, setSedeId] = useState<number | undefined>();
    const [data, setData] = useState<ResultadosData | null>(null);
    const [loading, setLoading] = useState(true);

    useEffect(() => {
        const fetchResultados = async () => {
            setLoading(true);
            try {
                const response = await adminApi.resultados(sedeId);
                setData(response.data);
                if (response.data.sedes) {
                    setSedes(response.data.sedes);
                }
            } catch (error) {
                console.error('Error fetching resultados:', error);
            } finally {
                setLoading(false);
            }
        };
        fetchResultados();
    }, [sedeId]);

    if (loading) {
        return (
            <div className="flex items-center justify-center h-64">
                <div className="spinner"></div>
            </div>
        );
    }

    if (!data) {
        return <div className="alert alert-error">Error al cargar los resultados</div>;
    }

    const totalVotosPersonero = data.personeros.reduce((acc, c) => acc + c.votos_count, 0);
    const totalVotosContralor = data.contralores.reduce((acc, c) => acc + c.votos_count, 0);

    const personeroChartData = {
        labels: data.chartDataPersoneros.labels,
        datasets: [{
            label: 'Votos',
            data: data.chartDataPersoneros.data,
            backgroundColor: ['#22c55e', '#4ade80', '#86efac', '#bbf7d0', '#dcfce7'],
            borderRadius: 8,
        }],
    };

    const contralorChartData = {
        labels: data.chartDataContralores.labels,
        datasets: [{
            label: 'Votos',
            data: data.chartDataContralores.data,
            backgroundColor: ['#eab308', '#facc15', '#fde047', '#fef08a', '#fef9c3'],
            borderRadius: 8,
        }],
    };

    const chartOptions = {
        responsive: true,
        maintainAspectRatio: false,
        indexAxis: 'y' as const,
        plugins: {
            legend: { display: false },
        },
        scales: {
            x: { beginAtZero: true },
        },
    };

    return (
        <div className="space-y-6">
            {/* Filtro por Sede */}
            <div className="card">
                <div className="flex items-center gap-4">
                    <label className="font-semibold text-gray-700">Filtrar por Sede:</label>
                    <select
                        value={sedeId || ''}
                        onChange={(e) => setSedeId(e.target.value ? Number(e.target.value) : undefined)}
                        className="form-input max-w-xs"
                    >
                        <option value="">Todas las Sedes</option>
                        {sedes.map(sede => (
                            <option key={sede.id} value={sede.id}>{sede.nombre}</option>
                        ))}
                    </select>
                </div>
            </div>

            {/* Resumen */}
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                <div className="card text-center">
                    <div className="text-3xl font-bold text-green-600">{data.personeros.length}</div>
                    <div className="text-sm text-gray-500">Candidatos Personero</div>
                </div>
                <div className="card text-center">
                    <div className="text-3xl font-bold text-yellow-600">{data.contralores.length}</div>
                    <div className="text-sm text-gray-500">Candidatos Contralor</div>
                </div>
                <div className="card text-center">
                    <div className="text-3xl font-bold text-blue-600">{totalVotosPersonero}</div>
                    <div className="text-sm text-gray-500">Votos Personero</div>
                </div>
                <div className="card text-center">
                    <div className="text-3xl font-bold text-purple-600">{totalVotosContralor}</div>
                    <div className="text-sm text-gray-500">Votos Contralor</div>
                </div>
            </div>

            {/* Gráficas */}
            <div className="grid md:grid-cols-2 gap-6">
                {/* Personero */}
                <div className="card">
                    <h3 className="font-bold text-gray-700 mb-4">🎓 Candidatos Personero</h3>
                    {data.personeros.length > 0 ? (
                        <div style={{ height: `${Math.max(200, data.personeros.length * 50)}px` }}>
                            <Bar data={personeroChartData} options={chartOptions} />
                        </div>
                    ) : (
                        <div className="text-center text-gray-400 py-8">No hay candidatos</div>
                    )}
                </div>

                {/* Contralor */}
                <div className="card">
                    <h3 className="font-bold text-gray-700 mb-4">📋 Candidatos Contralor</h3>
                    {data.contralores.length > 0 ? (
                        <div style={{ height: `${Math.max(200, data.contralores.length * 50)}px` }}>
                            <Bar data={contralorChartData} options={chartOptions} />
                        </div>
                    ) : (
                        <div className="text-center text-gray-400 py-8">No hay candidatos</div>
                    )}
                </div>
            </div>

            {/* Tablas de Resultados */}
            <div className="grid md:grid-cols-2 gap-6">
                {/* Tabla Personeros */}
                <div className="card">
                    <h3 className="font-bold text-gray-700 mb-4">🏆 Ranking Personeros</h3>
                    <table className="w-full">
                        <thead className="bg-green-50">
                            <tr>
                                <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">#</th>
                                <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Candidato</th>
                                <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Sede</th>
                                <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Votos</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-gray-100">
                            {data.personeros.length > 0 ? (
                                data.personeros.map((candidato, index) => (
                                    <tr key={candidato.id} className={index === 0 ? 'bg-green-50' : ''}>
                                        <td className="px-4 py-3">
                                            {index === 0 ? '🥇' : index === 1 ? '🥈' : index === 2 ? '🥉' : index + 1}
                                        </td>
                                        <td className="px-4 py-3 font-medium">{candidato.nombre}</td>
                                        <td className="px-4 py-3 text-sm text-gray-500">{candidato.sede?.nombre || '-'}</td>
                                        <td className="px-4 py-3 text-center">
                                            <span className="badge badge-green">{candidato.votos_count}</span>
                                        </td>
                                    </tr>
                                ))
                            ) : (
                                <tr>
                                    <td colSpan={4} className="px-4 py-8 text-center text-gray-400">
                                        No hay candidatos registrados
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>

                {/* Tabla Contralores */}
                <div className="card">
                    <h3 className="font-bold text-gray-700 mb-4">🏆 Ranking Contralores</h3>
                    <table className="w-full">
                        <thead className="bg-yellow-50">
                            <tr>
                                <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">#</th>
                                <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Candidato</th>
                                <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Sede</th>
                                <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Votos</th>
                            </tr>
                        </thead>
                        <tbody className="divide-y divide-gray-100">
                            {data.contralores.length > 0 ? (
                                data.contralores.map((candidato, index) => (
                                    <tr key={candidato.id} className={index === 0 ? 'bg-yellow-50' : ''}>
                                        <td className="px-4 py-3">
                                            {index === 0 ? '🥇' : index === 1 ? '🥈' : index === 2 ? '🥉' : index + 1}
                                        </td>
                                        <td className="px-4 py-3 font-medium">{candidato.nombre}</td>
                                        <td className="px-4 py-3 text-sm text-gray-500">{candidato.sede?.nombre || '-'}</td>
                                        <td className="px-4 py-3 text-center">
                                            <span className="badge badge-yellow">{candidato.votos_count}</span>
                                        </td>
                                    </tr>
                                ))
                            ) : (
                                <tr>
                                    <td colSpan={4} className="px-4 py-8 text-center text-gray-400">
                                        No hay candidatos registrados
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    );
}

