'use client';

import { useEffect, useState } from 'react';
import { adminApi } from '@/lib/api';

interface Sede {
    id: number;
    nombre: string;
    slug: string;
    activa: boolean;
    grados_count?: number;
    estudiantes_count?: number;
}

export default function SedesPage() {
    const [sedes, setSedes] = useState<Sede[]>([]);
    const [loading, setLoading] = useState(true);
    const [showModal, setShowModal] = useState(false);
    const [editingSede, setEditingSede] = useState<Sede | null>(null);
    const [formData, setFormData] = useState({ nombre: '', slug: '', activa: true });
    const [saving, setSaving] = useState(false);
    const [error, setError] = useState('');

    const fetchSedes = async () => {
        try {
            const response = await adminApi.getSedes();
            setSedes(response.data);
        } catch (error) {
            console.error('Error fetching sedes:', error);
        } finally {
            setLoading(false);
        }
    };

    useEffect(() => {
        fetchSedes();
    }, []);

    const generateSlug = (nombre: string) => {
        return nombre
            .toLowerCase()
            .normalize('NFD')
            .replace(/[\u0300-\u036f]/g, '')
            .replace(/[^a-z0-9]+/g, '-')
            .replace(/(^-|-$)/g, '');
    };

    const handleNombreChange = (nombre: string) => {
        setFormData(prev => ({
            ...prev,
            nombre,
            slug: editingSede ? prev.slug : generateSlug(nombre)
        }));
    };

    const openCreateModal = () => {
        setEditingSede(null);
        setFormData({ nombre: '', slug: '', activa: true });
        setError('');
        setShowModal(true);
    };

    const openEditModal = (sede: Sede) => {
        setEditingSede(sede);
        setFormData({ nombre: sede.nombre, slug: sede.slug, activa: sede.activa });
        setError('');
        setShowModal(true);
    };

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        if (!formData.nombre.trim() || !formData.slug.trim()) {
            setError('Nombre y slug son requeridos');
            return;
        }

        setSaving(true);
        setError('');

        try {
            if (editingSede) {
                await adminApi.updateSede(editingSede.id, formData);
            } else {
                await adminApi.createSede(formData);
            }
            setShowModal(false);
            fetchSedes();
        } catch (err: unknown) {
            const error = err as { response?: { data?: { message?: string } } };
            setError(error.response?.data?.message || 'Error al guardar');
        } finally {
            setSaving(false);
        }
    };

    const handleDelete = async (sede: Sede) => {
        if (!confirm(`¿Eliminar sede "${sede.nombre}"? Esta acción no se puede deshacer.`)) {
            return;
        }

        try {
            await adminApi.deleteSede(sede.id);
            fetchSedes();
        } catch (err: unknown) {
            const error = err as { response?: { data?: { message?: string } } };
            alert(error.response?.data?.message || 'Error al eliminar');
        }
    };

    if (loading) {
        return (
            <div className="flex items-center justify-center h-64">
                <div className="spinner"></div>
            </div>
        );
    }

    return (
        <div className="space-y-6">
            {/* Header */}
            <div className="flex justify-between items-center">
                <div>
                    <p className="text-gray-500">Administra las sedes de la institución</p>
                </div>
                <button onClick={openCreateModal} className="btn-primary flex items-center gap-2">
                    <span>➕</span> Nueva Sede
                </button>
            </div>

            {/* Tabla */}
            <div className="card">
                <table className="w-full">
                    <thead className="bg-gray-50">
                        <tr>
                            <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Nombre</th>
                            <th className="px-4 py-3 text-left text-sm font-semibold text-gray-600">Slug</th>
                            <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Grados</th>
                            <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Estudiantes</th>
                            <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Estado</th>
                            <th className="px-4 py-3 text-center text-sm font-semibold text-gray-600">Acciones</th>
                        </tr>
                    </thead>
                    <tbody className="divide-y divide-gray-100">
                        {sedes.length > 0 ? (
                            sedes.map(sede => (
                                <tr key={sede.id} className="hover:bg-gray-50">
                                    <td className="px-4 py-3 font-medium">{sede.nombre}</td>
                                    <td className="px-4 py-3 text-gray-500 font-mono text-sm">{sede.slug}</td>
                                    <td className="px-4 py-3 text-center">
                                        <span className="badge badge-blue">{sede.grados_count || 0}</span>
                                    </td>
                                    <td className="px-4 py-3 text-center">
                                        <span className="badge badge-green">{sede.estudiantes_count || 0}</span>
                                    </td>
                                    <td className="px-4 py-3 text-center">
                                        <span className={`badge ${sede.activa ? 'badge-green' : 'badge-red'}`}>
                                            {sede.activa ? 'Activa' : 'Inactiva'}
                                        </span>
                                    </td>
                                    <td className="px-4 py-3">
                                        <div className="flex items-center justify-center gap-2">
                                            <button
                                                onClick={() => openEditModal(sede)}
                                                className="p-2 text-blue-600 hover:bg-blue-50 rounded-lg transition"
                                                title="Editar"
                                            >
                                                ✏️
                                            </button>
                                            <button
                                                onClick={() => handleDelete(sede)}
                                                className="p-2 text-red-600 hover:bg-red-50 rounded-lg transition"
                                                title="Eliminar"
                                            >
                                                🗑️
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            ))
                        ) : (
                            <tr>
                                <td colSpan={6} className="px-4 py-8 text-center text-gray-400">
                                    No hay sedes registradas
                                </td>
                            </tr>
                        )}
                    </tbody>
                </table>
            </div>

            {/* Modal */}
            {showModal && (
                <div className="fixed inset-0 bg-black/50 flex items-center justify-center z-50">
                    <div className="bg-white rounded-2xl p-6 w-full max-w-md shadow-xl">
                        <h2 className="text-xl font-bold mb-4">
                            {editingSede ? 'Editar Sede' : 'Nueva Sede'}
                        </h2>

                        {error && <div className="alert alert-error mb-4">{error}</div>}

                        <form onSubmit={handleSubmit} className="space-y-4">
                            <div>
                                <label className="form-label">Nombre</label>
                                <input
                                    type="text"
                                    value={formData.nombre}
                                    onChange={(e) => handleNombreChange(e.target.value)}
                                    className="form-input"
                                    placeholder="Ej: Sede Principal"
                                    required
                                />
                            </div>

                            <div>
                                <label className="form-label">Slug (URL)</label>
                                <input
                                    type="text"
                                    value={formData.slug}
                                    onChange={(e) => setFormData(prev => ({ ...prev, slug: e.target.value }))}
                                    className="form-input font-mono"
                                    placeholder="sede-principal"
                                    required
                                />
                                <p className="text-xs text-gray-500 mt-1">Se usa en la URL: /sede/{formData.slug || 'slug'}</p>
                            </div>

                            <div className="flex items-center gap-3">
                                <input
                                    type="checkbox"
                                    id="activa"
                                    checked={formData.activa}
                                    onChange={(e) => setFormData(prev => ({ ...prev, activa: e.target.checked }))}
                                    className="w-5 h-5 rounded border-gray-300 text-green-600 focus:ring-green-500"
                                />
                                <label htmlFor="activa" className="text-sm font-medium text-gray-700">
                                    Sede activa
                                </label>
                            </div>

                            <div className="flex gap-3 pt-4">
                                <button
                                    type="button"
                                    onClick={() => setShowModal(false)}
                                    className="flex-1 px-4 py-3 border border-gray-300 rounded-xl hover:bg-gray-50 transition"
                                >
                                    Cancelar
                                </button>
                                <button
                                    type="submit"
                                    disabled={saving}
                                    className="flex-1 btn-primary disabled:opacity-50"
                                >
                                    {saving ? 'Guardando...' : 'Guardar'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            )}
        </div>
    );
}
