'use client';

import { useState } from 'react';
import Link from 'next/link';
import { authApi } from '@/lib/api';

export default function ForgotPasswordPage() {
    const [email, setEmail] = useState('');
    const [error, setError] = useState('');
    const [success, setSuccess] = useState('');
    const [loading, setLoading] = useState(false);

    const handleSubmit = async (e: React.FormEvent) => {
        e.preventDefault();
        setError('');
        setSuccess('');
        setLoading(true);

        try {
            const response = await authApi.forgotPassword({ email });
            setSuccess(response.data.message);
        } catch (err: unknown) {
            const errorMessage = err instanceof Error && 'response' in err
                ? (err as { response?: { data?: { message?: string } } }).response?.data?.message
                : 'Error al enviar el correo';
            setError(errorMessage || 'Error al enviar el correo');
        } finally {
            setLoading(false);
        }
    };

    return (
        <div
            className="min-h-screen flex items-center justify-center px-6"
            style={{ background: 'linear-gradient(135deg, #1B5E20 0%, #2E7D32 50%, #4CAF50 100%)' }}
        >
            <div className="w-full max-w-md">
                <div className="text-center mb-8">
                    <Link href="/">
                        <div className="w-24 h-24 mx-auto rounded-full bg-white p-2 shadow-lg flex items-center justify-center mb-4">
                            <span className="text-4xl">🏫</span>
                        </div>
                    </Link>
                    <h1 className="text-2xl font-bold text-white">Recuperar Contraseña</h1>
                    <p className="text-white/70">I.E. Las Huacas</p>
                </div>

                <div className="bg-white rounded-3xl p-8 shadow-2xl">
                    <p className="text-gray-600 text-sm mb-6 text-center">
                        Ingresa tu correo electrónico y te enviaremos un enlace para restablecer tu contraseña.
                    </p>

                    {error && <div className="alert alert-error mb-4">{error}</div>}
                    {success && <div className="alert alert-success mb-4">{success}</div>}

                    <form onSubmit={handleSubmit} className="space-y-5">
                        <div>
                            <label htmlFor="email" className="form-label">
                                Correo Electrónico
                            </label>
                            <input
                                type="email"
                                id="email"
                                value={email}
                                onChange={(e) => setEmail(e.target.value)}
                                className="form-input"
                                required
                                autoFocus
                            />
                        </div>

                        <button
                            type="submit"
                            className="btn-primary w-full py-3"
                            disabled={loading}
                        >
                            {loading ? 'Enviando...' : 'Enviar Enlace de Recuperación'}
                        </button>
                    </form>
                </div>

                <div className="text-center mt-6">
                    <Link href="/login" className="text-white/80 hover:text-white underline text-sm">
                        ← Volver al inicio de sesión
                    </Link>
                </div>
            </div>
        </div>
    );
}
