'use client';

import Link from 'next/link';
import { useEffect, useState } from 'react';
import { votacionApi } from '@/lib/api';

interface Sede {
  id: number;
  nombre: string;
  slug: string;
  activa: boolean;
}

interface Institucion {
  nombre: string;
  logo: string | null;
}

export default function Home() {
  const [sedes, setSedes] = useState<Sede[]>([]);
  const [votacionAbierta, setVotacionAbierta] = useState(true);
  const [institucion, setInstitucion] = useState<Institucion>({ nombre: 'I.E. Las Huacas', logo: null });
  const [loading, setLoading] = useState(true);

  const apiBaseUrl = process.env.NEXT_PUBLIC_API_URL?.replace('/api', '') || 'http://localhost:8000';

  useEffect(() => {
    const fetchSedes = async () => {
      try {
        const response = await votacionApi.getSedes();
        setSedes(response.data.sedes);
        setVotacionAbierta(response.data.votacionAbierta);
        if (response.data.institucion) {
          setInstitucion(response.data.institucion);
        }
      } catch (error) {
        console.error('Error fetching sedes:', error);
      } finally {
        setLoading(false);
      }
    };

    fetchSedes();
  }, []);

  return (
    <div className="min-h-screen flex items-center justify-center text-center px-6"
      style={{ background: 'linear-gradient(135deg, #1B5E20 0%, #2E7D32 50%, #4CAF50 100%)' }}>
      <div className="max-w-4xl">
        {/* Logo */}
        <div className="w-32 h-32 md:w-48 md:h-48 mx-auto mb-8 rounded-full bg-white p-2 shadow-lg flex items-center justify-center overflow-hidden">
          <img
            src={`${apiBaseUrl}/logo.jpg`}
            alt={institucion.nombre}
            className="w-full h-full object-contain"
            onError={(e) => {
              const target = e.target as HTMLImageElement;
              target.style.display = 'none';
              target.parentElement!.innerHTML = '<span class="text-4xl md:text-6xl">🏫</span>';
            }}
          />
        </div>

        {/* Title */}
        <h1 className="text-4xl md:text-7xl font-bold text-white mb-6"
          style={{ textShadow: '0 4px 20px rgba(0, 0, 0, 0.3)' }}>
          {institucion.nombre}
        </h1>

        <p className="text-xl md:text-2xl text-white/90 mb-12 max-w-2xl mx-auto">
          Sistema de Votación Estudiantil para Personero y Contralor
        </p>

        {/* Status Badge */}
        <div className="mb-8">
          {votacionAbierta ? (
            <span className="inline-flex items-center px-6 py-2 rounded-full text-lg font-semibold bg-green-100 text-green-800">
              🟢 Votación Abierta
            </span>
          ) : (
            <span className="inline-flex items-center px-6 py-2 rounded-full text-lg font-semibold bg-red-100 text-red-800">
              🔴 Votación Cerrada
            </span>
          )}
        </div>

        {/* Sedes Grid */}
        {loading ? (
          <div className="flex justify-center">
            <div className="spinner"></div>
          </div>
        ) : votacionAbierta ? (
          <div className="grid md:grid-cols-2 gap-6 mb-12">
            {sedes.map((sede) => (
              <Link
                key={sede.id}
                href={`/sede/${sede.slug}`}
                className="card hover:scale-105 transition-transform"
              >
                <h3 className="text-xl font-bold text-gray-800 mb-2">
                  🏛️ {sede.nombre}
                </h3>
                <p className="text-green-600">Click para votar →</p>
              </Link>
            ))}
          </div>
        ) : (
          <div className="mb-12">
            <Link
              href="/resultados"
              className="btn-secondary text-xl px-8 py-4 inline-block"
            >
              📊 Ver Resultados
            </Link>
          </div>
        )}
      </div>
    </div>
  );
}
