'use client';

import { useEffect, useState } from 'react';
import Link from 'next/link';
import { votacionApi } from '@/lib/api';

interface Candidato {
    id: number;
    nombre: string;
    foto?: string;
    votos_count: number;
}

interface SedeResultados {
    sede: { id: number; nombre: string; slug: string };
    personeros: Candidato[];
    contralores: Candidato[];
}

export default function ResultadosPage() {
    const [resultados, setResultados] = useState<SedeResultados[]>([]);
    const [loading, setLoading] = useState(true);
    const [error, setError] = useState('');

    useEffect(() => {
        const fetchResultados = async () => {
            try {
                const response = await votacionApi.resultadosPublicos();
                setResultados(response.data.resultados);
            } catch (err: unknown) {
                const errorMessage = err instanceof Error && 'response' in err
                    ? (err as { response?: { data?: { error?: string } } }).response?.data?.error
                    : 'Error al cargar resultados';
                setError(errorMessage || 'Los resultados aún no están disponibles');
            } finally {
                setLoading(false);
            }
        };

        fetchResultados();
    }, []);

    if (loading) {
        return (
            <div className="min-h-screen flex items-center justify-center"
                style={{ background: 'linear-gradient(135deg, #1B5E20 0%, #2E7D32 50%, #4CAF50 100%)' }}>
                <div className="spinner"></div>
            </div>
        );
    }

    if (error) {
        return (
            <div className="min-h-screen flex items-center justify-center text-center px-6"
                style={{ background: 'linear-gradient(135deg, #1B5E20 0%, #2E7D32 50%, #4CAF50 100%)' }}>
                <div className="max-w-md">
                    <div className="text-6xl mb-6">🕐</div>
                    <h1 className="text-3xl font-bold text-white mb-4">Resultados Pendientes</h1>
                    <p className="text-white/80 mb-8">{error}</p>
                    <Link href="/" className="btn-secondary">
                        Volver al Inicio
                    </Link>
                </div>
            </div>
        );
    }

    return (
        <div className="min-h-screen py-12"
            style={{ background: 'linear-gradient(135deg, #1B5E20 0%, #2E7D32 50%, #4CAF50 100%)' }}>
            <div className="container mx-auto px-6">
                {/* Header */}
                <div className="text-center mb-12">
                    <Link href="/">
                        <div className="w-24 h-24 mx-auto rounded-full bg-white p-2 shadow-lg flex items-center justify-center mb-4">
                            <span className="text-4xl">🏫</span>
                        </div>
                    </Link>
                    <h1 className="text-4xl font-bold text-white mb-2">Resultados de la Votación</h1>
                    <p className="text-white/80">I.E. Las Huacas</p>
                </div>

                {/* Results by Sede */}
                <div className="grid md:grid-cols-2 gap-8 max-w-6xl mx-auto">
                    {resultados.map((sedeData) => (
                        <div key={sedeData.sede.id} className="card">
                            <h2 className="text-2xl font-bold text-gray-800 mb-6 text-center border-b pb-4">
                                🏛️ {sedeData.sede.nombre}
                            </h2>

                            <div className="grid md:grid-cols-2 gap-6">
                                {/* Personeros */}
                                <div>
                                    <h3 className="font-bold text-green-700 mb-4 text-center">🎓 Personero</h3>
                                    {sedeData.personeros.map((p, index) => (
                                        <div
                                            key={p.id}
                                            className={`p-4 rounded-xl mb-3 ${index === 0 ? 'bg-gradient-to-r from-green-100 to-green-50 border-2 border-green-400' : 'bg-gray-50'}`}
                                        >
                                            <div className="flex items-center gap-3">
                                                <div className={`w-12 h-12 rounded-full flex items-center justify-center text-white font-bold ${index === 0 ? 'bg-gradient-to-br from-green-500 to-green-600' : 'bg-gray-400'}`}>
                                                    {index === 0 ? '🏆' : index + 1}
                                                </div>
                                                <div className="flex-1">
                                                    <div className="font-bold text-gray-800">{p.nombre}</div>
                                                    <div className={`text-sm font-semibold ${index === 0 ? 'text-green-600' : 'text-gray-500'}`}>
                                                        {p.votos_count} votos
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>

                                {/* Contralores */}
                                <div>
                                    <h3 className="font-bold text-yellow-700 mb-4 text-center">📋 Contralor</h3>
                                    {sedeData.contralores.map((c, index) => (
                                        <div
                                            key={c.id}
                                            className={`p-4 rounded-xl mb-3 ${index === 0 ? 'bg-gradient-to-r from-yellow-100 to-yellow-50 border-2 border-yellow-400' : 'bg-gray-50'}`}
                                        >
                                            <div className="flex items-center gap-3">
                                                <div className={`w-12 h-12 rounded-full flex items-center justify-center text-white font-bold ${index === 0 ? 'bg-gradient-to-br from-yellow-500 to-yellow-600' : 'bg-gray-400'}`}>
                                                    {index === 0 ? '🏆' : index + 1}
                                                </div>
                                                <div className="flex-1">
                                                    <div className="font-bold text-gray-800">{c.nombre}</div>
                                                    <div className={`text-sm font-semibold ${index === 0 ? 'text-yellow-600' : 'text-gray-500'}`}>
                                                        {c.votos_count} votos
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </div>
                        </div>
                    ))}
                </div>

                {/* Back Link */}
                <div className="text-center mt-12">
                    <Link href="/" className="text-white/80 hover:text-white underline text-sm">
                        ← Volver a la página principal
                    </Link>
                </div>
            </div>
        </div>
    );
}
