import axios from 'axios';

const api = axios.create({
    baseURL: process.env.NEXT_PUBLIC_API_URL || 'http://localhost:8000/api',
    headers: {
        'Content-Type': 'application/json',
        'Accept': 'application/json',
    },
    withCredentials: false, // Usar solo autenticación basada en tokens, no cookies/sesiones
});

// Request interceptor to add auth token
api.interceptors.request.use((config) => {
    if (typeof window !== 'undefined') {
        const token = localStorage.getItem('token');
        if (token) {
            config.headers.Authorization = `Bearer ${token}`;
        }
    }
    return config;
});

// Response interceptor to handle errors
api.interceptors.response.use(
    (response) => response,
    (error) => {
        if (error.response?.status === 401) {
            if (typeof window !== 'undefined') {
                localStorage.removeItem('token');
                localStorage.removeItem('user');
                // Redirect to login only if not already on login page
                if (!window.location.pathname.includes('/login')) {
                    window.location.href = '/login';
                }
            }
        }
        return Promise.reject(error);
    }
);

export default api;

// Auth API
export const authApi = {
    login: (data: { email: string; password: string }) =>
        api.post('/login', data),
    register: (data: { name: string; email: string; password: string; password_confirmation: string }) =>
        api.post('/register', data),
    logout: () =>
        api.post('/logout'),
    user: () =>
        api.get('/user'),
    forgotPassword: (data: { email: string }) =>
        api.post('/forgot-password', data),
    resetPassword: (data: { token: string; email: string; password: string; password_confirmation: string }) =>
        api.post('/reset-password', data),
};

// Admin API
export const adminApi = {
    dashboard: () => api.get('/admin/dashboard'),
    toggleVotacion: () => api.post('/admin/toggle-votacion'),
    resultados: (sedeId?: number) =>
        api.get('/admin/resultados', { params: sedeId ? { sede_id: sedeId } : {} }),

    // Sedes
    getSedes: () => api.get('/admin/sedes'),
    createSede: (data: { nombre: string; slug: string; activa?: boolean }) =>
        api.post('/admin/sedes', data),
    updateSede: (id: number, data: { nombre: string; slug: string; activa?: boolean }) =>
        api.put(`/admin/sedes/${id}`, data),
    deleteSede: (id: number) => api.delete(`/admin/sedes/${id}`),

    // Grados
    getGrados: () => api.get('/admin/grados'),
    getGradosBySede: (sedeId: number) => api.get(`/admin/sedes/${sedeId}/grados`),
    createGrado: (data: { nombre: string; sede_id: number }) =>
        api.post('/admin/grados', data),
    updateGrado: (id: number, data: { nombre: string; sede_id: number }) =>
        api.put(`/admin/grados/${id}`, data),
    deleteGrado: (id: number) => api.delete(`/admin/grados/${id}`),

    // Candidatos
    getCandidatos: (filters?: { sede_id?: number; tipo?: string }) =>
        api.get('/admin/candidatos', { params: filters }),
    createCandidato: (data: FormData) =>
        api.post('/admin/candidatos', data, { headers: { 'Content-Type': 'multipart/form-data' } }),
    updateCandidato: (id: number, data: FormData) =>
        api.post(`/admin/candidatos/${id}`, data, { headers: { 'Content-Type': 'multipart/form-data' } }),
    deleteCandidato: (id: number) => api.delete(`/admin/candidatos/${id}`),

    getEstudiantes: (filters?: { grado_id?: number; sede_id?: number; habilitado?: boolean }) =>
        api.get('/admin/estudiantes', { params: filters }),
    createEstudiante: (data: { name: string; email?: string; documento: string; grado_id: number; habilitado?: boolean }) =>
        api.post('/admin/estudiantes', data),
    updateEstudiante: (id: number, data: { name: string; documento: string; grado_id: number; habilitado?: boolean }) =>
        api.put(`/admin/estudiantes/${id}`, data),
    deleteEstudiante: (id: number) => api.delete(`/admin/estudiantes/${id}`),
    habilitarTodos: () => api.post('/admin/estudiantes/habilitar-todos'),
    resetearVotos: () => api.post('/admin/estudiantes/resetear-votos'),
    exportEstudiantes: (filters?: { sede_id?: number; grado_id?: number }) =>
        api.get('/admin/estudiantes-export', { params: filters, responseType: 'blob' }),
    importEstudiantes: (data: FormData) =>
        api.post('/admin/estudiantes-import', data, { headers: { 'Content-Type': 'multipart/form-data' } }),
};

// Votacion API (public)
export const votacionApi = {
    getSedes: () => api.get('/sedes'),
    getSede: (slug: string) => api.get(`/sede/${slug}`),
    verificar: (slug: string, documento: string) =>
        api.post(`/sede/${slug}/verificar`, { documento }),
    getCandidatos: (slug: string, token: string) =>
        api.post(`/sede/${slug}/candidatos`, { token }),
    votar: (slug: string, data: { token: string; personero_id?: number; contralor_id?: number }) =>
        api.post(`/sede/${slug}/votar`, data),
    resultadosPublicos: () => api.get('/resultados-publicos'),
};
